/*** 
 * 
 * @author Dominik Schindler 
 * 
 * Instrumenter.java - Main class to instrument the input file.
 * 
 * Java source -> JavaLexer & JavaRecognizer -> AST -> JavaPreInstrumenter -> 
 * JavaTreeInstrumenter -> instrumented AST & symbol table -> JavaTreeWriter -> 
 * instrumented Java source  
 *
 */

package de.fau.cs.swe.sa.conditionCoverage;

import java.io.*;

import java.awt.Dimension;
import java.awt.Toolkit;

import antlr.*;
import antlr.collections.AST;
import antlr.debug.misc.ASTFrame;

public class Instrumenter {
	
	// save symbolCount = id 
	private long symbolCount = 0;
	private AST parserAST;
	private AST instrAST;
	private FileOutputStream symbolTable;

	// instrument file without class hierarchie (= used for pc)
	public void instrumentFile( File inFile, File outFile ) throws IOException, FileNotFoundException, TokenStreamException, RecognitionException {
				
		// use linereader for line number
		LineNumberReader fis = new LineNumberReader( new BufferedReader (  new FileReader ( inFile ) ) );
		FileOutputStream fos = new FileOutputStream( outFile );
		
		try {
			JavaLexer lexer = new JavaLexer( fis );
			JavaRecognizer parser = new JavaRecognizer( lexer );
			parser.compilationUnit( fis );
		
			JavaTreePreInstrumenter preinstr = new JavaTreePreInstrumenter();
			preinstr.compilationUnit( parser.getAST() );
		
			JavaTreeInstrumenter instr = new JavaTreeInstrumenter( preinstr.getVariableDecls(), preinstr.getMethodDecls(), preinstr.getImports() );
			instr.setSymbolCount( symbolCount );
			parserAST = parser.getAST();
			instr.compilationUnit( parserAST );
			instrAST = instr.getAST();
			symbolCount = instr.getSymbolCount();
			
			JavaTreeWriter writer = new JavaTreeWriter( fos );
			writer.setSymbolTable( symbolTable );
			writer.compilationUnit( instr.getAST() );
			
		} finally {
			fis.close();
			fos.close();
		}
	}

	// instrument file with class hierarchie (=pc)
	public void instrumentFile( File inFile, File outFile, ClassHierarchie ch ) throws IOException, FileNotFoundException, TokenStreamException, RecognitionException {
				
		// use linereader for line number
		LineNumberReader fis = new LineNumberReader( new BufferedReader (  new FileReader ( inFile ) ) );
		FileOutputStream fos = new FileOutputStream( outFile );
		
		try {
			JavaLexer lexer = new JavaLexer( fis );
			JavaRecognizer parser = new JavaRecognizer( lexer );
			parser.compilationUnit( fis );
			
			JavaTreePreInstrumenter preinstr = new JavaTreePreInstrumenter();
			preinstr.compilationUnit( parser.getAST() );			
			preinstr.printMethodDecls();
		
			JavaTreeInstrumenter instr = new JavaTreeInstrumenter( ch, preinstr.getVariableDecls(), preinstr.getMethodDecls(), preinstr.getImports() );
			instr.setSymbolCount( symbolCount );
			parserAST = parser.getAST();
			instr.compilationUnit( parserAST );
			instrAST = instr.getAST();
			symbolCount = instr.getSymbolCount();
			
			JavaTreeWriter writer = new JavaTreeWriter( fos ); //, ch, preinstr.getVariableDecls(), preinstr.getMethodDecls() );
			writer.setSymbolTable( symbolTable );
			writer.compilationUnit( instr.getAST() );
			
		} finally {
			fis.close();
			fos.close();
		}
	}	
	
	public Instrumenter( FileOutputStream symbolTable ) {
		this.symbolTable = symbolTable;
	}

	public void closeSymbolTable() throws IOException {
		symbolTable.close();
	}
	
	public AST getAST() {
		return parserAST;
	}
	
	public AST getInstrAST() {
		return instrAST;
	}
	
	public static void showAST ( File f ) throws FileNotFoundException, TokenStreamException, RecognitionException {
		FileInputStream fis = new FileInputStream(f);
		
		JavaLexer lexer = new JavaLexer( fis );
		JavaRecognizer parser = new JavaRecognizer( lexer );
		parser.compilationUnit( null );
		
		ASTFactory factory = new ASTFactory();
		AST r = factory.create( 0,"AST ROOT" );
		r.setFirstChild( parser.getAST() );
		
		ASTFrame frame = new ASTFrame( "Java AST of \"" + f.getPath() + "\"", r );
		frame.setSize( new Dimension( 800, 600 ) );

		Dimension d = Toolkit.getDefaultToolkit().  getScreenSize()  ;
		frame.setLocation( ( d.width - frame.getSize().width ) / 2, ( d.height - frame.getSize().height ) / 2 );
		
		frame.setVisible( true );
	}

	public static void showAST ( AST a ) throws FileNotFoundException, TokenStreamException, RecognitionException {
		ASTFactory factory = new ASTFactory();
		AST r = factory.create( 0,"AST ROOT" );
		r.setFirstChild(a);
		
		ASTFrame frame = new ASTFrame("Java AST", r);
		frame.setSize( new Dimension( 800, 600 ) );

		Dimension d = Toolkit.getDefaultToolkit().getScreenSize()  ;
		frame.setLocation( ( d.width - frame.getSize().width ) / 2, ( d.height - frame.getSize().height ) / 2 );
		
		frame.setVisible( true );
	}
	
	public static int getMajorVersion() {
		return 0;
	}
	
	public static int getMinorVersion() {
		return 11;
	}
	
	public static String getSupportedJavaVersion() {
		return "1.5";
	}
	
}