/*** 
 * 
 * @author Dominik Schindler 
 * 
 * ClassHierarchie.java - Creates and manages the class hierarchie.
 * 
 */

package de.fau.cs.swe.sa.conditionCoverage;

import java.io.*;
import java.util.*;
import java.net.*;
import java.util.jar.*;
import java.util.zip.*;

public class ClassHierarchie {
	
	// define constants
	public URLClassLoader theURLClassLoader;
	private Vector classPaths;
	private Hashtable knownClassesHashtable = new Hashtable();

	public void setupKnownClassesHashtable() throws DuplicateClassException, IOException, ClassNotFoundException {
		
		String myClassName;
		Class myClass;
		ExtendedClass myExtendedClass;
		
		// put wrapper classes for primitive types
		knownClassesHashtable.put( Boolean.TYPE.getName(), new ExtendedClass(Boolean.TYPE ));
		knownClassesHashtable.put( Character.TYPE.getName(), new ExtendedClass(Character.TYPE) );
		knownClassesHashtable.put( Byte.TYPE.getName(), new ExtendedClass(Byte.TYPE) );
		knownClassesHashtable.put( Short.TYPE.getName(), new ExtendedClass(Short.TYPE) );
		knownClassesHashtable.put( Integer.TYPE.getName(), new ExtendedClass(Integer.TYPE) );
		knownClassesHashtable.put( Long.TYPE.getName(), new ExtendedClass(Long.TYPE) );
		knownClassesHashtable.put( Float.TYPE.getName(), new ExtendedClass(Float.TYPE) );
		knownClassesHashtable.put( Double.TYPE.getName(), new ExtendedClass(Double.TYPE) );
		knownClassesHashtable.put( Void.TYPE.getName(), new ExtendedClass(Void.TYPE) );
		
		for ( int classPathEntryIndex = 0; classPathEntryIndex < classPaths.size(); classPathEntryIndex++ ) {
			
			ClassPath c =  ( ClassPath ) classPaths.get( classPathEntryIndex );
			
			System.out.println( "Reading: " + c.f + "..." );
			
			if ( c.f.isDirectory() ) {
				
				File[] filesInDirectory = SystemUtils.getAllFilesInDirectory( c.f, SystemUtils.CLASS_FILENAME_EXTENSION);
				String rootName = c.f.getCanonicalPath();

				for ( int fileIndex = 0; fileIndex < filesInDirectory.length; fileIndex++ ) {
					
					myClassName = filesInDirectory[ fileIndex ].getCanonicalPath();
					myClassName = myClassName.substring( rootName.length() + 1, myClassName.length() - SystemUtils.CLASS_FILENAME_EXTENSION.length() );
					myClassName = myClassName.replace( File.separatorChar, SystemUtils.PACKAGE_SEPARATOR_CHAR );

	System.out.println( "-" + c.pack + "- " + myClassName );
					
					if ( c != null && c.pack != null && !c.pack.equals("") ) {
						myClassName = c.pack + "." + myClassName;  			
					}
					
	System.out.println( "-" + myClassName + "-" );
				
	
					myClass = Class.forName( myClassName, false, theURLClassLoader );
					
					myClassName = myClass.getName();
					
					
					
					if ( !knownClassesHashtable.contains( myClassName ) ) {
						myExtendedClass = new ExtendedClass( myClass );
						knownClassesHashtable.put( myClassName, myExtendedClass );
					} else {
						throw new DuplicateClassException ("Duplicate class: " + myClassName);
					}
				}
			} else if ( c.f.isFile() ) {
				
				JarFile jarFile = new JarFile( c.f );
				Enumeration entries = jarFile.entries();
				
				while( entries.hasMoreElements() ) {
					
					ZipEntry zipEntry = (ZipEntry)entries.nextElement();
					myClassName = zipEntry.getName(); 
					if ( !zipEntry.isDirectory() && myClassName.toLowerCase().endsWith( SystemUtils.CLASS_FILENAME_EXTENSION ) ) {
						myClassName = myClassName.substring(0, myClassName.length() - SystemUtils.CLASS_FILENAME_EXTENSION.length());
						myClassName = myClassName.replace( SystemUtils.ZIP_PATH_SEPARATOR_CHAR, SystemUtils.PACKAGE_SEPARATOR_CHAR );
						myClass = Class.forName( myClassName, false, theURLClassLoader );
						myClassName = myClass.getName();

						if ( !knownClassesHashtable.contains( myClassName ) ) {
							myExtendedClass = new ExtendedClass( myClass );
							knownClassesHashtable.put( myClassName, myExtendedClass );
						} else {
							throw new DuplicateClassException ( "Duplicate class: " + myClassName );
						}
					}
				}
			}
		}
	}

	public void setupInheritanceRelations() throws ClassNotFoundException {
		ExtendedClass myExtendedClass, myExtendedSuperClass, myExtendedSuperInterface;
		Class myClass, mySuperClass, mySuperInterfaces[];
		String mySuperClassName, mySuperInterfaceName;
		
		Enumeration knownClasses = knownClassesHashtable.elements();
		
		while( knownClasses.hasMoreElements() ) {
			myExtendedClass = ( ExtendedClass ) knownClasses.nextElement();
			myClass = myExtendedClass.getThisClass();
			mySuperClass = myClass.getSuperclass();
			
			if ( mySuperClass != null ) {
				mySuperClassName = mySuperClass.getName();
				
				myExtendedSuperClass = ( ExtendedClass ) knownClassesHashtable.get( mySuperClassName ) ;
				//if ( isClassAccessible( myClass ) ) {
					if ( !myExtendedClass.getSubClasses().contains( myExtendedClass ) ) {
						myExtendedSuperClass.addSubClass( myExtendedClass );
					}
				//}
			}
			
			mySuperInterfaces = myClass.getInterfaces();
			
			for ( int interfaceIndex = 0; interfaceIndex < mySuperInterfaces.length; interfaceIndex++ ) {
				mySuperInterfaceName = mySuperInterfaces[interfaceIndex].getName();
				myExtendedSuperInterface = ( ExtendedClass ) knownClassesHashtable.get( mySuperInterfaceName );
				
				//if ( isClassAccessible(myClass) ) {
					if ( !myExtendedSuperInterface.getSubClasses().contains( myExtendedClass ) ) {
						myExtendedSuperInterface.addSubClass( myExtendedClass );
					}
				//}
			}
		}
	}	
	
	public ExtendedClass getExtendedClass( String s ) {
		return ( ExtendedClass ) knownClassesHashtable.get( s );
	}
	
	public void addClassPath ( File f, String p ) {
		classPaths.add( new ClassPath( f, p ) );
	}
	
	public void printKnownClassesHastable() {
		
		Enumeration enu = knownClassesHashtable.elements();
		
		while( enu.hasMoreElements() ) {
			ExtendedClass myExtendedClass = ( ExtendedClass ) enu.nextElement();

			System.out.println ( myExtendedClass.toString() );
		}
	}
	
	
	public ClassHierarchie() {
		// get all locations for classes
		StringTokenizer st = new StringTokenizer( System.getProperty( "sun.boot.class.path" ), System.getProperty( "path.separator" ) );
	
		classPaths = new Vector();

		int i = 0;		
		while ( st.hasMoreElements() ) {
			String s = st.nextToken();
			classPaths.add( new ClassPath ( new File ( s ), null ) );
			i++;
		}
		
		// add current working directory to classpath
		try {
			URL[] url = new URL[1];
			url[0] = new File( System.getProperty( "user.dir" ) ).toURL();
			theURLClassLoader = new URLClassLoader( url );
		} catch ( Exception e) {
			System.out.println( e );
		}
	}
	
	public void addClassSearchPath( File f ) {
		
		URL[] urls = new URL[ theURLClassLoader.getURLs().length + 1 ];
		
		for ( int i = 0; i < theURLClassLoader.getURLs().length; i++ ) {
			urls[ i ] = theURLClassLoader.getURLs()[ i ];
		}
		
		try {
			urls[ theURLClassLoader.getURLs().length ] = f.toURL();		
		} catch ( MalformedURLException mue ) {
			urls = theURLClassLoader.getURLs();
		}
		
		theURLClassLoader = new URLClassLoader ( urls, ClassLoader.getSystemClassLoader() );		
	}

	public void addClassAndSearchPath( File classPath, String packageName, File searchPath  ) {
		addClassPath( classPath, packageName );
		addClassSearchPath( searchPath );		
	}
	
	public void printURLs() {
		for ( int i = 0; i < theURLClassLoader.getURLs().length; i++ ) {
			System.out.println( theURLClassLoader.getURLs()[ i ] );
		}
	}
	
	public static void main ( String args[] ) {
		
		ClassHierarchie ch = null;
		
		ch = new ClassHierarchie();
		ch.addClassAndSearchPath( new File ( "C:/Dokumente und Einstellungen/root/Desktop/in/samples/" ), "samples", new File ( "C:/Dokumente und Einstellungen/root/Desktop/in/" ) );
		
		ch.printURLs();
		
		try {
			ch.setupKnownClassesHashtable();
		} catch ( DuplicateClassException dce ) {
			System.out.println( "1:" + dce );			
		} catch ( IOException ioe ) {
			System.out.println( "1:" + ioe );			
		} catch ( ClassNotFoundException cnfe ) {
			System.out.println( "1:" + cnfe );
		}
		
		try {
			ch.setupInheritanceRelations();
		} catch ( Exception e) {
			System.out.println( "2:" + e );
		}

		//ch.printKnownClassesHastable();
		
		ExtendedClass e = ch.getExtendedClass( "samples.BigFib" );
		
		System.out.println ( e.getThisClass().getName() );
		System.out.println ( e.getThisClass().getSuperclass().getName() );

	}
}

class DuplicateClassException extends Exception {
	public DuplicateClassException ( String s ) {
		super ( s );
	}
	
}

class ClassPath {
	public File f;
	public String pack;
	public ClassPath ( File f, String pack) {
		this.f = f;
		this.pack = pack;
	}
}

