/*** 
 * 
 * @author Dominik Schindler 
 * 
 * Analyzer.java - Main analyzer class.
 * 
 * Methods:
 * 		loadSymbolTable( String ): Loads the symbol table for analyzing
 * 		addLogfile( File ): Adds a logfile to be analyzed 
 *  	clear(): Clear symbol table and logfiles list
 *  	printSymbolTable(): Prints the symbol table debug output to stdout
 *  	removeLogfile( File ): Removes a logfile
 *  	removeLogfile( int ): Remove a logfile
 * 		getOverallSCC(): Returns simple condition coverage for all conditions and
 * 						 testcases (= logfiles) as CoverageDetails object
 *		getOverallCDC(): Returns condition/decition coverage -"-
 *		getOverallMMCC(): Returns modified multiple condition coverage -"-
 *		getOverallMCDC(): Returns modified condition/decition coverage -"-
 *		getOverallMCC(): Returns multiple condition coverage -"-
 *		getSCC( Symbol ): Returns simple condition coverage for a combined / branch
 *						  condition, method or class as CoverageDetails object
 *		getCDC( Symbol ): Returns condition/decition coverage -"- 
 * 		getMMCC( Symbol ): Returns modified multiple condition coverage -"-
 * 		getMCDC( Symbol ): Returns modified condition/decition coverage -"-
 * 		getMCC( Symbol ): Returns multiple condition coverage -"-
 * 
 */

package de.fau.cs.swe.sa.conditionCoverage;

import java.io.*;

// Import utils classes
import java.util.*;
import java.util.regex.*;

public class Analyzer {
	
		private Vector Logfiles = null;
		private Hashtable SymbolTable = null;
		
		public void loadSymbolTable( File f ) throws IOException, AnalyzerException {
			
			BufferedReader in;
			Vector combined = new Vector();
			Vector branch = new Vector();
			
			// uses regex and "\t" as separator of symbol table entries
			Pattern p = Pattern.compile("[\t]");
			
			// read the symbol table
			in = new BufferedReader( new FileReader( f ) );
		      
	    	String line = in.readLine();

			// variables to store the ids used later 
	    	long methodId = -1;
			long classId  = -1;
			long switchId = -1;
			long variableId = -1;
			
			while ( line != null ) {
			    // use regex to split the line into:
		    	// row[0] = ID, row[1] = Row, row[2] = Type, row[3] = Expression
			    // row[4] = Notes (optional) 
				String[] rows = p.split(line); 

		    	long id 	 = Long.parseLong(rows[0]);
		    	long row	 = Long.parseLong(rows[1]);
		    	String type  = rows[2];
		    	String exp 	 = rows[3];
		    	String notes;
		    	
		    	// determine, if the current line has some notes (currently empty)
		    	if (rows.length == 4 + 1) {
			    	notes = rows[4];
    		  	} else if ( rows.length == 3 + 1 ) {
    		  		notes = "";
    		  	} else {
	    			throw new AnalyzerException( "Symbol table corrupt as there are not 3 or 4 columns!" );
    		  	}

		    	// if current symbol = class, set current classId to symbolId
		    	if ( type.equals( "ke" ) || type.equals( "kl" ) ) {
		    		// no difference between kl and ke 
		    		type = new String( "k" );
		    		classId = id;
		    	}
		    	
		    	// if current symbol = method, set current methodId to symbolId
		    	if ( type.equals( "m" ) || type.equals( "o" ) ) {
		    		methodId = id;
		    	}

		    	// if current symbol = switch, set current switchId to symbolId
		    	if ( type.equals( "w" ) ) {
		    		switchId = id;
		    	}
		    	
		    	// if current symbol = variable, set current variable id to symbolId
		    	if ( type.equals( "v" ) ) {
		    		variableId = id;
		    	}
		    	  	
		    	Symbol sym = new Symbol( id, row, type, exp, notes, classId, methodId, -1, -1 );
			    SymbolTable.put( new Long( sym.id ), sym ); // ID ==> symbol

			    // === some adjustments after sym creation ===

			    // if current symbol = case, create a link to the current switchId
		    	if ( type.equals( "s" ) ) {
		    		sym.belongsToId = switchId;
		    	}
			    
			    // store all conditons belonging to a brach or combined condition
		    	// for further processing
			    if ( type.equals( "a" ) || type.equals( "p" ) || 
			    	type.equals( "c" ) ) {
			    	branch.add( sym );
			    	combined.add( sym );
			    }

		    	// if current symbol = branch, create a link from all previously stored conditions to   
			    // the id of the branch
			    if ( type.equals( "b" ) ) {
		    		Symbol s;
		    		for ( int i = 0; i < branch.size(); i++ ) {
		    			s = ( Symbol ) branch.get( i );
		    			s.belongsToId = sym.id;
		    		}
		    		// clear lists for next branch
		    		branch.clear();
		    		combined.clear();
		    	}
		    	
		    	// if current symbol = combined condition, set combinedId of all previously
			    // stored symbols to the id of the combined condition
		    	if ( type.equals( "c" ) ) {
		    		Symbol s;
		    		for ( int i = 0; i < combined.size(); i++ ) {
		    			s = ( Symbol ) combined.get( i );
		    			s.conditionId.add( new Long ( sym.id ));
		    		}
		    	}  	
		    	
		    	// if current symbol = class/interface, set variableId 
		    	if ( type.equals( "i" ) ) {
		    		sym.belongsToId = variableId;
		    	}
		    	
			    line = in.readLine();
		    }
		}

		// for debug output
		public void printSymbolTable() {
			System.out.println("ID\tRow\tT-Count\tF-Count\tTYPE\tC-ID\tM-ID\tCd-IDs\tBel-Id\tVar-Id\tEXP");
			Symbol sym;
	       
			Enumeration symbols = SymbolTable.elements();
			while ( symbols.hasMoreElements() ) {
	        	sym = (Symbol) symbols.nextElement();
		    	System.out.println( 
		    			sym.id + "\t" + 
		    			sym.row + "\t" + 
		    			sym.trueCount + "\t" + 
		    			sym.falseCount + "\t" + 
		    			sym.type + "\t" + 
		    			sym.classId + "\t" +
		    			sym.methodId + "\t" +
		    			sym.conditionId + "\t" +
		    			sym.belongsToId + "\t" + 
		    			sym.exp);
	        } 	
		}

		public Hashtable getSymbolTable() {
			return SymbolTable;
		}
		
		public Vector getLogfiles() {
			return Logfiles;
		}
		
		// === logfile methods === 
		public void addLogfile( File f ) {
			Logfiles.add( new LogfileItem( f, true ) );
		}
		
		public void removeLogfile( int index ) {
			Logfiles.remove( index );
		}
		
		public void removeLogfile( File f ) {
			Logfiles.remove( f );
		}
		
		public void clear() {
			SymbolTable.clear();
			Logfiles.clear();
		}
		
		// === coverage methods ===
		// === overall coverages ===
		
		// overall simple condition coverage
		public CoverageDetails getOverallSCC() throws IOException {
			
			// temporary vector
			Vector v = new Vector();
			Symbol sym;
			
	        // iterate through symbol table ...
			Enumeration enu = SymbolTable.elements();
				
	        while ( enu.hasMoreElements() ) {
	        	sym = ( Symbol ) enu.nextElement();
	        	// ... and select the atomar and primary expressions
	        	if ( sym.type.equals( "a" ) || sym.type.equals( "p" ) ) {
	        		v.add( sym );
	        	}
	        } 	
			
			return getCoverage(v);
		}
		
		// overall condition-/decition coverage
		public CoverageDetails getOverallCDC() throws Exception {
			
			// temporary vector
			Vector v = new Vector();
			Symbol sym;
			
	        // iterate through symbol table ...
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	sym = ( Symbol ) enu.nextElement();
	        	// ... and select the atomar, primary and branach expressions
	        	if ( sym.type.equals( "a" ) || sym.type.equals( "p" ) || sym.type.equals( "b" )) {
	        		v.add( sym );
	        	}
	        } 	
			
			return getCoverage(v);
		}
		
		// overall multiple condition coverage
		public CoverageDetails getOverallMMCC() throws Exception {
			
			// temporary vector
			Vector v = new Vector();
			Symbol sym;
			
	        // iterate through symbol table ...
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	sym = ( Symbol ) enu.nextElement();
	        	// ... and select the atomar, primary, branch and combined expressions
	        	if ( sym.type.equals( "a" ) || sym.type.equals( "p" ) || sym.type.equals( "b" ) || sym.type.equals( "c" )) {
	        		v.add( sym );
	        	}
	        } 	
			return getCoverage(v);
		}
		
		// overall modified condition coverage
		public OverallCoverage getOverallMCDC() throws IOException {
			
			Symbol sym;
			OverallCoverage oc = new OverallCoverage();
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	sym = ( Symbol ) enu.nextElement();
	        	// ... and select branach conditions
	        	if ( sym.type.equals( "b" ) ) {
	        		MCDCoverageDetails mcdcod = getMCDC( sym );
	        		Double percentage = new Double( ( double ) mcdcod.getCovered().size() / mcdcod.getRows() );
	        		oc.addSymbol( sym, percentage );
	        	}
	        } 	
	        
			return oc;
		}	
		
		// overall multiple condition coverage
		public OverallCoverage getOverallMCC() throws IOException {

			Symbol sym;
			OverallCoverage oc = new OverallCoverage();
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	sym = ( Symbol ) enu.nextElement();
	        	// ... and select branach conditions
	        	if ( sym.type.equals( "b" ) ) {
	        		MCCoverageDetails mccod = getMCC( sym );
	        		Double percentage = new Double( ( double ) mccod.getCovered().size() / mccod.getRows() );
	        		oc.addSymbol( sym, percentage );
	        	}
	        } 	
	        
			return oc;
		}	

		// overall method coverage (for overloading)
		public CoverageDetails getOverallMC() throws Exception {
			
			// temporary vector
			Vector v = new Vector();
			Symbol sym;
			
	        // iterate through symbol table ...
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	sym = ( Symbol ) enu.nextElement();
	        	// ... and select the methods and constructors
	        	if ( sym.type.equals( "m" ) || sym.type.equals( "o" )) {
	        		v.add( sym );
	        	}
	        } 	
			
			return getMethodCoverage( v );
		}
		
		// overall switch coverage
		public CoverageDetails getOverallSC() throws Exception {
			
			// temporary vector
			Vector v = new Vector();
			Symbol sym;
			
	        // iterate through symbol table ...
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	sym = ( Symbol ) enu.nextElement();
	        	// ... and select the cases
	        	if ( sym.type.equals( "s" ) ) {
	        		v.add( sym );
	        	}
	        } 	
			
			return getCoverage(v);
		}
		
		// overall polymophic condition coverage
		public OverallCoverage getOverallPC() throws IOException {
			
			Symbol sym;
			OverallCoverage oc = new OverallCoverage();
	        // iterate through symbol table ...
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	sym = ( Symbol ) enu.nextElement();
	        	// ... and select the variables
	        	if ( sym.type.equals( "v" ) ) {
	        		CoverageDetails cod = getPC( sym );
	        		Double percentage = new Double( ( double ) cod.getTrueOnly() / cod.getCount() );
	        		oc.addSymbol( sym, percentage );
	        	}
	        } 	
	        
			return oc;
		}	
	
		// === finegrained coverage for ids ===
		// simple condition coverage
		public CoverageDetails getSCC(Symbol sym) throws Exception {
			if (sym == null) return null;
			
			// temporary vector for simple conditions
			Vector v = new Vector();
			Symbol tempSymbol;
			
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	tempSymbol = ( Symbol ) enu.nextElement();

	        	// ... and select the symbols for scc only
	        	if (!( tempSymbol.type.equals( "a" ) || tempSymbol.type.equals( "p" ) ) ) {
	        		continue;
	        	}
	        	// ignore same symbol	
	        	if ( tempSymbol == sym ) continue;
	        	
	        	// if symbol = method, select the symbols with methodId = Id
	        	if ( sym.type.equals( "m" ) && ( tempSymbol.methodId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = ctor, select the symbols with methodId = Id
	        	if ( sym.type.equals( "o" ) && ( tempSymbol.methodId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = class, select the symbols with classId = Id
	        	if ( sym.type.equals( "k" ) && ( tempSymbol.classId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = combined conditions, select the symbols with conditionId = Id
	        	if ( sym.type.equals( "c" ) && ( tempSymbol.conditionId.contains( new Long ( sym.id ) ) ) ) { 
	        		v.add( tempSymbol );
	        		// System.out.println( sym + " " + tempSymbol );
	        	}
	        	// if symbol = branch, select the symbols with branchId = Id
	        	if ( sym.type.equals( "b" ) && ( tempSymbol.belongsToId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        } 	

	        if (v.size() > 0) {
				return getCoverage(v);
			} else {
				return null;
			}
		}
		
		// condition decition coverage 
		public CoverageDetails getCDC(Symbol sym) throws Exception {	
			if (sym == null) return null;

			// temporary vector for conditions
			Vector v = new Vector();
			Symbol tempSymbol;
			
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	tempSymbol = ( Symbol ) enu.nextElement();

	        	// ... and select the symbols for cdc only
	        	if ( !( tempSymbol.type.equals( "a" ) || tempSymbol.type.equals( "p" ) ||
	        			tempSymbol.type.equals ("t") || tempSymbol.type.equals( "b" ) ) ) {
	        		continue;
	        	}
	        	// ignore same symbol	
	        	if ( tempSymbol == sym ) continue;
	        	
	        	// if symbol = method, select the symbols with methodId = Id
	        	if ( sym.type.equals( "m" ) && ( tempSymbol.methodId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = ctor, select the symbols with methodId = Id
	        	if ( sym.type.equals( "o" ) && ( tempSymbol.methodId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = class, select the symbols with classId = Id
	        	if ( sym.type.equals( "k" ) && ( tempSymbol.classId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = combined conditions, select the symbols with conditionId = Id
	        	if ( sym.type.equals( "c" ) && ( tempSymbol.conditionId.contains( new Long( sym.id ) ) ) ) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = branch, select the symbols with branchId = Id
	        	if ( sym.type.equals( "b" ) && ( tempSymbol.belongsToId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        } 	

	        if (v.size() > 0) {
				return getCoverage(v);
			} else {
				return null;
			}
		}
		
		// minimal multiple cc 
		public CoverageDetails getMMCC(Symbol sym) throws Exception {
			if (sym == null) return null;

			// temporary vector for conditions
			Vector v = new Vector();
			Symbol tempSymbol;
			
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	tempSymbol = ( Symbol ) enu.nextElement();

	        	// ... and select the symbols for cdc only
	        	if (!( tempSymbol.type.equals( "a" ) || tempSymbol.type.equals( "p" ) ||
	        			tempSymbol.type.equals ("t") || tempSymbol.type.equals( "b" ) ||
	        			tempSymbol.type.equals( "b" ))) {
	        		continue;
	        	}
	        	// ignore same symbol	
	        	if ( tempSymbol == sym ) continue;
	        	
	        	// if symbol = method, select the symbols with methodId = Id
	        	if ( sym.type.equals( "m" ) && ( tempSymbol.methodId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = ctor, select the symbols with methodId = Id
	        	if ( sym.type.equals( "o" ) && ( tempSymbol.methodId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = class, select the symbols with classId = Id
	        	if ( sym.type.equals( "k" ) && ( tempSymbol.classId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = combined conditions, select the symbols with conditionId = Id
	        	if ( sym.type.equals( "c" ) && ( tempSymbol.conditionId.contains( new Long ( sym.id ) ) ) ) { 
	        		v.add( tempSymbol );
	        	}
	        	// if symbol = brach, select the symbols with branchId = Id
	        	if ( sym.type.equals( "b" ) && ( tempSymbol.belongsToId == sym.id)) { 
	        		v.add( tempSymbol );
	        	}
	        } 	

	        if (v.size() > 0) {
				return getCoverage(v);
			} else {
				return null;
			}
		}

		// modified condition/decition cov
		public MCDCoverageDetails getMCDC(Symbol sym) throws IOException {
			if (sym == null) return null;

			// temporary vector for conditions
			Vector v = new Vector();
			Symbol tempSymbol;
			
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	tempSymbol = ( Symbol ) enu.nextElement();

	        	// ... and select the symbols for mcc only
	        	if ( ! ( tempSymbol.type.equals( "a" ) || tempSymbol.type.equals( "p" ) ) ) {
	        		continue;
	        	}
	        	// ignore same symbol	
	        	if ( tempSymbol == sym ) continue;
	        	
	        	// if symbol = brach, select the symbols with branchId = Id
	        	if ( tempSymbol.belongsToId == sym.id) { 
	        		v.add( new Long ( tempSymbol.id ) );
	        	}
	        } 	
	        
	        if (v.size() > 0) {
				return getMCDCoverage( v, sym );
			} else {
				return null;
			}
		}	
		
		// multiple condition cov
		public MCCoverageDetails getMCC( Symbol sym ) throws IOException {
			if (sym == null) return null;

			// temporary vector for conditions
			Vector v = new Vector();
			Symbol tempSymbol;
			
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	tempSymbol = ( Symbol ) enu.nextElement();

	        	// ... and select the symbols for mcc only
	        	if ( ! ( tempSymbol.type.equals( "a" ) || tempSymbol.type.equals( "p" ) ) ) {
	        		continue;
	        	}
	        	// ignore same symbol	
	        	if ( tempSymbol == sym ) continue;
	        	
	        	// if symbol = branch, select the symbols with branchId = Id
	        	if ( tempSymbol.belongsToId == sym.id) { 
	        		v.add( new Long ( tempSymbol.id ) );
	        	}
	        } 	
	        
	        if (v.size() > 0) {
				return getMCCoverage( v, sym );
			} else {
				return null;
			}
		}	
		
		// mc
		public CoverageDetails getMC(Symbol sym) throws Exception {
			if (sym == null) return null;

			// temporary vector for conditions
			Vector v = new Vector();
			Symbol tempSymbol;
			
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	tempSymbol = ( Symbol ) enu.nextElement();

	        	// ... and select the symbols for mc only (methods & constructors )
	        	if (!( tempSymbol.type.equals( "m" ) || tempSymbol.type.equals( "o" ) ) ) {
	        		continue;
	        	}

	        	// ignore same symbol	
	        	// if ( tempSymbol == sym ) continue;
	        	
	        	// extract method head
	        	int p = sym.exp.indexOf( ' ' );
	        	int q = sym.exp.indexOf( '(' );
	        	int index = Math.min( p, q );
	        	String methodName = sym.exp.substring( 0, index);
	        	
	        	// if symbol = combined condition, select the symbols with conditionId = Id
	        	if ( tempSymbol.exp.indexOf ( methodName ) >= 0 ) { 
	        		v.add( tempSymbol );
	        	}
	        } 	

	        if ( v.size() > 0 ) {
				return getMethodCoverage( v );
			} else {
				return null;
			}		
	    }	

		// switch coverage
		public CoverageDetails getSC(Symbol sym) throws Exception {
			if (sym == null) return null;

			// temporary vector for conditions
			Vector v = new Vector();
			Symbol tempSymbol;
			
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	tempSymbol = ( Symbol ) enu.nextElement();

	        	// ... and select the symbols for sc only
	        	if (!( tempSymbol.type.equals( "s" ) ) ) {
	        		continue;
	        	}

	        	// ignore same symbol	
	        	if ( tempSymbol == sym ) continue;
	        	
	        	// if symbol = switch, select the cases which belong to the switch
	        	if ( tempSymbol.belongsToId ==  sym.id ) { 
	        		v.add( tempSymbol );
	        	}
	        } 	

	        if (v.size() > 0) {
				return getCoverage(v);
			} else {
				return null;
			}		
	    }	
		
		// polymorphic coverage
		public CoverageDetails getPC( Symbol sym ) throws IOException {
			if (sym == null) return null;

			// temporary vector for conditions
			Vector v = new Vector();
			Symbol tempSymbol;
			
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	tempSymbol = ( Symbol ) enu.nextElement();

	        	// ... and select the symbols for pc only
	        	if ( ! ( tempSymbol.type.equals( "i" ) ) ) {
	        		continue;
	        	}
	        	// ignore same symbol	
	        	if ( tempSymbol == sym ) continue;
	        	
	        	// if symbol = variable, select the types which belong to the variable
	        	if ( tempSymbol.belongsToId == sym.id) { 
	        		v.add( tempSymbol );
	        	}
	        } 	
	        
	        if (v.size() > 0) {
				return getCoverage( v );
			} else {
				return null;
			}			
		}
		

		// generic coverage computation fr scc, cdc, mmcc
		public CoverageDetails getCoverage( Vector symbols ) throws IOException {
			
			if (symbols == null) return null;

			// clear all variant symbol attributes
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	Symbol sym = ( Symbol ) enu.nextElement();
	        	sym.clear();
	       	}			
			
			LoggerInputStream log;
			LogfileTriple lft;
			CoverageDetails cod = new CoverageDetails();
			
			for ( int i = 0; i < Logfiles.size(); i++ ) {

		        Stack startEndPairs = new Stack();
				LogfileItem lfi = ( LogfileItem ) Logfiles.get( i );
				
				if ( !lfi.isSelected() ) continue;
				
				File f = lfi.getFile();
				
				log = new LoggerInputStream( new FileInputStream( f ) );
				
		    	while ( ( lft = log.readLine() ) != null ) {	    	  
	                
	                // enter method
	                if ( lft.id == LoggerTypes.ENTER_METHOD ) {
		    			startEndPairs.push( new StartEndPairsElement ( LoggerTypes.ENTER_METHOD, lft.value  ) );
	                	continue;
	                }
	                
	                // left method
	                if ( lft.id == LoggerTypes.LEFT_METHOD ) {
		    			//startEndPairs.push( new StartEndPairsElement ( LoggerTypes.LEFT_METHOD, lft.value  ) );
	                	continue;
	                }
		    		
		    		// treat special ids
	                if ( lft.id == LoggerTypes.START_EXPRESSION ) {
		    			startEndPairs.push( new StartEndPairsElement ( LoggerTypes.START_EXPRESSION, lft.value  ) );
	                	continue;
	                }

		    		// end of expression
		    		if ( lft.id == LoggerTypes.END_EXPRESSION ) {
			    		Symbol sym = ( Symbol ) SymbolTable.get( new Long( lft.value ) );

		    			// check, if start and end is equal and remove pair from stack
			    		if ( !startEndPairs.empty() ) {
		    				StartEndPairsElement e = ( StartEndPairsElement ) startEndPairs.peek();
		    				
		    				if ( e.type == LoggerTypes.START_EXPRESSION && e.id == lft.value ) {
		    					startEndPairs.pop();
		    				} else {
		    					sym.unknownCount++;
		    				}
		    			}
	                }
	                
		    		// if handle expression called before endxpression, expression not valid
	                if ( lft.id == LoggerTypes.HANDLE_EXCEPTION ) {
 
	        			if ( !startEndPairs.empty() ) {
		    				StartEndPairsElement e = ( StartEndPairsElement ) startEndPairs.peek();
				    		Symbol sym = ( Symbol ) SymbolTable.get( new Long( e.id ) );
		    				
		    				// "end expression" not executed, as there is a "start expression"
				    		// left
				    		if ( e.type == LoggerTypes.START_EXPRESSION ) {
	    						sym.unknownCount++;
	    						
	    						// if an exception occured, remove the "enter method" from stack
	    						startEndPairs.pop();
	    						e = ( StartEndPairsElement ) startEndPairs.peek();
	    						
	    						if ( e.type == LoggerTypes.ENTER_METHOD ) {
	    							startEndPairs.pop();
			    					// System.out.println("Enter method removed!");
	    						}
		    				} else if ( e.type == LoggerTypes.ENTER_METHOD ) {
		    					// in this method has no condition, remove "enter method" from stack
		    					startEndPairs.pop();
		    					// System.out.println("Method with no condition!");
		    				}
		    			}
						continue;
	                }
	                
	                // ignore not treated special ids
	                if ( lft.id < 0 ) continue;
	                
		    		Symbol sym = ( Symbol ) SymbolTable.get( new Long( lft.id ) );
	                
		    		try {
		    			if ( lft.value == LoggerTypes.TRUE_VALUE ) {
		    				// 	increase the trueCount for the id
		    				sym.trueCount++;
		    			} else if ( lft.value == LoggerTypes.FALSE_VALUE ) {
		    				// increase the falseCount for the id
		    				sym.falseCount++;
		    			} else if ( lft.value == LoggerTypes.UNKNOWN_VALUE ) {
		    				// increase unknownCount for the id
		    				sym.unknownCount++;
		    			} else {
		    				// boolean value can only be true or false!
		    				throw new IOException ( "Logfile corrupt! Truth value (" + lft.value + ") has not been defined!" ); 
		    			}
		    		} catch ( NullPointerException e ) {
		    			// Symbol with id not found in HashMap
		    			throw new IOException ( "Element with ID = " + lft.id + " not found in symbol table.\nLogfile or symbol table corrupt or don't match!" );
		    		}
		    	}				
		    	
		    	log.close();
		    	
            	// System.out.println ( "END: " + startEndPairs );
			
			}	
			
			// add symbols to the coverage lists
			for ( int i = 0; i < symbols.size(); i++ ) {
				Symbol sym = ( Symbol ) symbols.get( i );
				if (( sym.trueCount > 0 ) && ( sym.falseCount > 0 )) {
					cod.addCovered(sym);
				} else if ( sym.trueCount > 0 ) {
					cod.addTrueOnly(sym);
				} else if ( sym.falseCount > 0 ) {
					cod.addFalseOnly(sym);
				} else if ( sym.unknownCount > 0 ) {
					cod.addUnknwon(sym);
				} 
				if ( sym.trueCount == 0 && sym.falseCount == 0 && sym.unknownCount == 0 ) {
					cod.addNever(sym);
				}
			}
			
			return cod;
		}
		// multiple condition coverage
		public MCCoverageDetails getMCCoverage( Vector symbols, Symbol sym ) throws IOException {
			
			if (symbols == null );
			LoggerInputStream log;
			LogfileTriple lft;
			MCCoverageDetails mmccod = new MCCoverageDetails( symbols );
			Vector trueConditions = new Vector();
			Vector falseConditions = new Vector();
			Vector notCoveredConditions = new Vector();
			
			for ( int i = 0; i < Logfiles.size(); i++ ) {
				
				Stack startEndPairs = new Stack();
				LogfileItem lfi = ( LogfileItem ) Logfiles.get( i );
				if ( !lfi.isSelected() ) continue;
				File f = lfi.getFile();
				log = new LoggerInputStream( new FileInputStream( f ) );

		    	while ( ( lft = log.readLine() ) != null ) {	    	  
		    		
		    		// see getCoverage() 
		    		if ( lft.id == LoggerTypes.ENTER_METHOD ) {
		    			startEndPairs.push( new StartEndPairsElement( LoggerTypes.ENTER_METHOD, lft.value ) );
		    			continue;
		    		}

	                // see getCoverage()
	                if ( lft.id == LoggerTypes.LEFT_METHOD ) {
		    			//startEndPairs.push( new StartEndPairsElement ( LoggerTypes.LEFT_METHOD, lft.value  ) );
	                	continue;
	                }
	                
		    		// start of expression
		    		if ( lft.id == LoggerTypes.START_EXPRESSION && lft.value == sym.id ) {

		    			startEndPairs.push( new StartEndPairsElement ( LoggerTypes.START_EXPRESSION, lft.value ) );
		    			trueConditions.clear();
		    			falseConditions.clear();
		    			notCoveredConditions = new Vector( symbols );
		    			continue;
	                }
	                		
		    		// end of expression
	                if ( lft.id == LoggerTypes.END_EXPRESSION  ) {

	                	if ( !startEndPairs.empty() ) {
	                		StartEndPairsElement pair = ( StartEndPairsElement ) startEndPairs.peek();
	                		
	                		if ( pair.type == LoggerTypes.START_EXPRESSION && pair.id == lft.value ){
	                			startEndPairs.pop();

	    				    	Integer[] b = new Integer[ symbols.size() ];
	    				    	// iterate through list and set rows 
	    				    	for( int j = 0; j < symbols.size(); j++ ) {
	    			                // if current symbol does not belong to the analyzed expression
	    			                // ignore
	    				    		if ( trueConditions.contains( symbols.get( j ) ) ) {
	    				    			b[j] = new Integer( 1 );
	    				    			System.out.print( b[j] );
	    				    		} else if ( falseConditions.contains( symbols.get( j ) ) ) {
	    				    			b[j] = new Integer( 0 );
	    				    			System.out.print( b[j] );
	    				    		} else if ( notCoveredConditions.contains( symbols.get( j) ) ) {
	    				    			// if all conditions are not covered, no wildcard
	    				    			// as it would falsely mark all rows!
	    				    			if ( notCoveredConditions.size() != symbols.size() ) {
	    				    				b[j] = new Integer( 2 );
	    				    			} else {
	    				    				b[j] = new Integer( 0 );
	    				    			}
	    				    			// System.out.print( b[j] );
	    				    		}
	    				    	}
	    				    	
	    				    	// mark line in table
	    						mmccod.mark( b );
	    						
	    		    			// System.out.println();
	    				    	
	    		    			trueConditions.clear();
	    		    			falseConditions.clear();
	    		    			notCoveredConditions = new Vector( symbols );
	                		} else {
	                			sym.unknownCount++;
	                		}
	                	}
	                	continue;
	                }
	                
		    		// if handle expression called before endxpression, expression not valid
	                if ( lft.id == LoggerTypes.HANDLE_EXCEPTION ) {
 
	        			if ( !startEndPairs.empty() ) {
		    				StartEndPairsElement e = ( StartEndPairsElement ) startEndPairs.peek();
		    				
		    				// "end expression" not executed, as there is a "start expression"
				    		// left
				    		if ( e.type == LoggerTypes.START_EXPRESSION ) {
	    						sym.unknownCount++;
	    						
	    						// if an exception occured, remove the "enter method" from stack
	    						startEndPairs.pop();
	    						e = ( StartEndPairsElement ) startEndPairs.peek();
	    						
	    						if ( e.type == LoggerTypes.ENTER_METHOD ) {
	    							startEndPairs.pop();
			    					// System.out.println("Enter method removed!");
	    						}
		    				} else if ( e.type == LoggerTypes.ENTER_METHOD ) {
		    					// in this method was no condition, remove "enter method" from stack
		    					startEndPairs.pop();
		    					// System.out.println("Method with no condition!");
		    				}
		    			}
						continue;
	                }	                
	                // ignore not treated special ids
	                if ( lft.id < 0 ) continue;

	                // add the symbol to the appropriate list and remove
	                // it from the notCovered list
	                Long l = new Long ( lft.id );
	                if ( !symbols.contains( l ) ) continue;

	                if ( lft.value == LoggerTypes.TRUE_VALUE ) {
	                	notCoveredConditions.remove( l );
	                	trueConditions.add( l );
	                } else  if ( lft.value == LoggerTypes.FALSE_VALUE) {
	                	notCoveredConditions.remove( l );
	                	falseConditions.add( l );
	                }
	                
			    	// System.out.println( trueConditions + "    " + falseConditions + "    " + notCoveredConditions );

		    	}
		    	
		    	log.close();
		    	
			}	
		
			return mmccod;
		}
		// modified condition-/decition coverage
		public MCDCoverageDetails getMCDCoverage( Vector symbols, Symbol sym ) throws IOException {
			
			if (symbols == null );
			LoggerInputStream log;
			LogfileTriple lft;
			MCDCoverageDetails mcdcod = new MCDCoverageDetails( symbols );
			Vector trueConditions = new Vector();
			Vector falseConditions = new Vector();
			Vector notCoveredConditions = new Vector();
			boolean evaluation = false;
			
			for ( int i = 0; i < Logfiles.size(); i++ ) {
				
				Stack startEndPairs = new Stack();
				LogfileItem lfi = ( LogfileItem ) Logfiles.get( i );
				if ( !lfi.isSelected() ) continue;
				File f = lfi.getFile();
				log = new LoggerInputStream( new FileInputStream( f ) );
			   
		    	while ( ( lft = log.readLine() ) != null ) {	    	  
		    		
		    		// see getCoverage()
		    		if ( lft.id == LoggerTypes.ENTER_METHOD ) {
		    			startEndPairs.push( new StartEndPairsElement( LoggerTypes.ENTER_METHOD, lft.value ) );
		    			continue;
		    		}

	                // see getCoverage()
	                if ( lft.id == LoggerTypes.LEFT_METHOD ) {
		    			//startEndPairs.push( new StartEndPairsElement ( LoggerTypes.LEFT_METHOD, lft.value  ) );
	                	continue;
	                }
	                
		    		// start of expression
		    		if ( lft.id == LoggerTypes.START_EXPRESSION && lft.value == sym.id ) {

		    			startEndPairs.push( new StartEndPairsElement ( LoggerTypes.START_EXPRESSION, lft.value ) );
		    			trueConditions.clear();
		    			falseConditions.clear();
		    			notCoveredConditions = new Vector( symbols );
		    			continue;
	                }
	                		
		    		// end of expression
	                if ( lft.id == LoggerTypes.END_EXPRESSION  ) {

	                	if ( !startEndPairs.empty() ) {
	                		StartEndPairsElement pair = ( StartEndPairsElement ) startEndPairs.peek();
	                		
	                		if ( pair.type == LoggerTypes.START_EXPRESSION && pair.id == lft.value ){
	                			startEndPairs.pop();

	    				    	Integer[] b = new Integer[ symbols.size() ];
	    				    	// iterate through list and set rows 
	    				    	for( int j = 0; j < symbols.size(); j++ ) {
	    			                // if symbol read does not contain to the analyzed expression
	    			                // ignore
	    				    		if ( trueConditions.contains( symbols.get( j ) ) ) {
	    				    			b[j] = new Integer( 1 );
	    				    			System.out.print( b[j] );
	    				    		} else if ( falseConditions.contains( symbols.get( j ) ) ) {
	    				    			b[j] = new Integer( 0 );
	    				    			System.out.print( b[j] );
	    				    		} else if ( notCoveredConditions.contains( symbols.get( j) ) ) {
	    				    			// if all conditions are not covered, no wildcard
	    				    			// as it would falsely marks all rows!
	    				    			if ( notCoveredConditions.size() != symbols.size() ) {
	    				    				b[j] = new Integer( 2 );
	    				    			} else {
	    				    				b[j] = new Integer( 0 );
	    				    			}
	    				    			// System.out.print( b[j] );
	    				    		}
	    				    	}
	    				    	
	    						mcdcod.addEvaluation( b, evaluation );
	    						
	    		    			//System.out.println();
	    				    	
	    		    			trueConditions.clear();
	    		    			falseConditions.clear();
	    		    			notCoveredConditions = new Vector( symbols );
	                		} else {
	                			sym.unknownCount++;
	                		}
	                	}
	                	continue;
	                }
	                
		    		// if handle expression called before endxpression, expression not valid
	                if ( lft.id == LoggerTypes.HANDLE_EXCEPTION ) {
 
	        			if ( !startEndPairs.empty() ) {
		    				StartEndPairsElement e = ( StartEndPairsElement ) startEndPairs.peek();
		    				
		    				// "end expression" not executed, as there is a "start expression"
				    		// left
				    		if ( e.type == LoggerTypes.START_EXPRESSION ) {
	    						sym.unknownCount++;
	    						
	    						// if an exception occured, remove the "enter method" from stack
	    						startEndPairs.pop();
	    						e = ( StartEndPairsElement ) startEndPairs.peek();
	    						
	    						if ( e.type == LoggerTypes.ENTER_METHOD ) {
	    							startEndPairs.pop();
			    					System.out.println("Enter method removed!");
	    						}
		    				} else if ( e.type == LoggerTypes.ENTER_METHOD ) {
		    					// in this method was no condition, remove "enter method" from stack
		    					startEndPairs.pop();
		    					System.out.println("Method with no condition!");
		    				}
		    			}
						continue;
	                }	                
	                
	                // ignore not treated special ids
	                if ( lft.id < 0 ) continue;

	                // if id is evalution, store overall evalution
	                if ( lft.id == sym.id ) {
	                	evaluation = ( lft.value == LoggerTypes.TRUE_VALUE );
	                	continue;
	                }
	                
	                // add the symbol to the appropriate list and remove
	                // it from the notCovered list
	                Long l = new Long ( lft.id );
	                if ( !symbols.contains( l ) ) continue;

	                if ( lft.value == LoggerTypes.TRUE_VALUE ) {
	                	notCoveredConditions.remove( l );
	                	trueConditions.add( l );
	                } else if ( lft.value == LoggerTypes.FALSE_VALUE) {
	                	notCoveredConditions.remove( l );
	                	falseConditions.add( l );
	                }
	                
			    	System.out.println( trueConditions + "    " + falseConditions + "    " + notCoveredConditions );

		    	}
		    	
		    	log.close();
		    	
			}
			
			mcdcod.compute();
			//mcdcod.printValuesTable();
			return mcdcod;
		}
		
		// method coverage
		public CoverageDetails getMethodCoverage( Vector symbols ) throws IOException {
			
			if (symbols == null) return null;

			// clear all variant symbol attributes
			Enumeration enu = SymbolTable.elements();
			
	        while ( enu.hasMoreElements() ) {
	        	Symbol sym = ( Symbol ) enu.nextElement();
	        	sym.clear();
	       	}			
			
			LoggerInputStream log;
			LogfileTriple lft;
			CoverageDetails cod = new CoverageDetails();
			
			for ( int i = 0; i < Logfiles.size(); i++ ) {
				
				LogfileItem lfi = ( LogfileItem ) Logfiles.get( i );
				
				if ( !lfi.isSelected() ) continue;
				
				File f = lfi.getFile();
		
				log = new LoggerInputStream( new FileInputStream( f ) );
			   
		    	while ( ( lft = log.readLine() ) != null ) {	    	  
	                // treat special ids
	                if ( lft.id == LoggerTypes.ENTER_METHOD ) {
			    		Symbol sym = ( Symbol ) SymbolTable.get( new Long( lft.value ) );
			    		// excecuted methods are covered
	                	if ( sym != null ) sym.trueCount++;
	                	continue;
	                }
		    	}
			}
			
			// add symbols to the coverage lists
			for (int i = 0; i < symbols.size(); i++) {
				Symbol sym = (Symbol)symbols.get(i);
				if ( ( sym.trueCount > 0 ) && ( sym.falseCount > 0 ) ) {
					cod.addCovered(sym);
				} else if ( sym.trueCount > 0 ) {
					cod.addTrueOnly(sym);
				} else if ( sym.falseCount > 0 ) {
					cod.addFalseOnly(sym);
				} else if ( sym.unknownCount > 0 ) {
					cod.addUnknwon(sym);
				} 
				if ( sym.trueCount == 0 && sym.falseCount == 0 && sym.unknownCount == 0 ) {
					cod.addNever(sym);
				}
			}
			
			return cod;
		}
				
		public Symbol getSymbolById ( long id ) {
			return ( Symbol ) SymbolTable.get( new Long ( id ) ) ;
		}
		
		public static int getMajorVersion() {
			return 0;
		}
		
		public static int getMinorVersion() {
			return 11;
		}
		
		// === ctors ===
		
		private void initilize() {
			SymbolTable = new Hashtable();
			Logfiles = new Vector();
		}
		
		public Analyzer() {
			initilize();
		}
		
}

class StartEndPairsElement {
	public long id;
	public long type;
	
	public StartEndPairsElement( long type, long id ) {
		this.id = id;
		this.type = type;
	}
	
	public String toString() {
		return id + " : " + type;
	}
	
}