package de.fau.cs.swe.da.test;

import static org.junit.Assert.*;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.omg.PortableInterceptor.ACTIVE;

import org.eclipse.emf.common.util.EList;
import org.eclipse.uml2.uml.*;

import de.cnc.expression.exceptions.ExpressionEvaluationException;
import de.cnc.expression.exceptions.ExpressionParseException;
import de.fau.cs.swe.da.exceptions.NoValidStateMachineException;
import de.fau.cs.swe.da.modelsimulator.*;
import de.fau.cs.swe.da.modelsimulator.MyEvent.EventType;
import de.fau.cs.swe.da.modelsimulator.Simulator.EventsQueueBehavior;
import de.fau.cs.swe.da.modelsimulator.Simulator.StopReason;

import java.io.*;
import java.util.*;

public class SimulatorTest {

	private static Model model = null;
	private static Simulator simulator = null;
	private static StateMachine stateMachine = null;
	private static SimulateableStateMachine simSM = null;
	
	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		ModelLoader modelLoader = new ModelLoader( new File( "modelle/Diplomarbeit.uml2" ) );
		model = modelLoader.getModel();
	
		Hashtable<String,StateMachine> stateMachines = SimulateableStateMachine.getStateMachines( model );
		
		stateMachine = stateMachines.get( "Data::Testmodelle::Autoradio::Autoradio" );	
		simSM = SimulateableStateMachine.getSimulateableStateMachine( model, stateMachine );
		simulator = new Simulator( simSM );
	}

	@AfterClass
	public static void tearDownAfterClass() throws Exception {
	}

	@Test
	public void testSimulator() throws ExpressionEvaluationException {
		assertNotNull( model );
		assertNotNull( stateMachine );
		assertNotNull( simSM );
		assertNotNull( simulator );
	}

	@Test
	public void testSetupSimulation() throws NoValidStateMachineException, ExpressionEvaluationException, ExpressionParseException {
		simulator.setupSimulation();
	}

	@Test
	public void testSetGetVariable() throws ExpressionEvaluationException {
		simulator.setVariable( "CD_drin", new Boolean( true ) );
		
		Object value = simulator.getVariable( "CD_drin" );
		if ( value instanceof Boolean ) {
			Boolean bool = ( Boolean ) value;
			assertTrue( bool.booleanValue() );
		} else {
			fail( "Invalid type of CD_drin!" );
		}
		
		assertNotNull( simulator.getVariableValues() );
		assertNotNull( simulator.getVariableTypes() );
		
		assertTrue( simulator.getVariableValues().size() > 0 );
		assertTrue( simulator.getVariableTypes().size() > 0 );
		
		simulator.setVariable( "CD_drin", new Boolean( false ) );
		
		value = simulator.getVariable( "CD_drin" );
		if ( value instanceof Boolean ) {
			Boolean bool = ( Boolean ) value;
			assertFalse( bool.booleanValue() );
		} else {
			fail( "Invalid type of CD_drin!" );
		}		
		
	}

	@Test
	public void testAddEvent() {
		
		ArrayList<MyEvent> events = simSM.getAvailableEvents();
		simulator.addEvents( events );
		
		assertTrue ( simulator.getActiveEvents().size() == simSM.getAvailableEvents().size() );
		
		for ( int i = simulator.getActiveEvents().size() - 1; i > -1; i-- ) {
			simulator.removeActiveEvent( i );
		}
		
		System.out.println( simulator.getActiveEvents() );
		
		assertTrue ( simulator.getActiveEvents().size() == 0 );
	}

	@Test
	public void testCallOperation() throws NoValidStateMachineException, ExpressionParseException, ExpressionEvaluationException {

		simulator.doStep();
		simulator.doStep();
		simulator.doStep();
		simulator.doStep();

		ArrayList<MyEvent> events = simSM.getAvailableEvents();
		
		MyEvent call_CD_eingelegt = null;
		MyEvent call_Radio_manuell = null;
		MyEvent call_CD_manuell = null;
		for ( int i = 0; i < events.size(); i++ ) {
			MyEvent myEvent = events.get( i );
			if ( myEvent.getEventType() == EventType.callOperation ) {
				Operation operation = myEvent.getOperation();
				if ( operation.getName().equalsIgnoreCase( "CD_eingelegt" ) ) {
					call_CD_eingelegt = simulator.callOperation( operation );
				}
				if ( operation.getName().equalsIgnoreCase( "Radio_manuell" ) ) {
					call_Radio_manuell = simulator.callOperation( operation );
				}
				if ( operation.getName().equalsIgnoreCase( "CD_manuell" ) ) {
					call_CD_manuell = simulator.callOperation( operation );
				}
			}
		}
		
		Parameter parameterB = call_CD_eingelegt.getOperation().getOwnedParameter( "b", null );
		simulator.setParameterValue( call_CD_eingelegt, parameterB, "true" );
		
		assertTrue( simulator.getActiveEvents().size() == 3 );

		simulator.doStep();
		assertTrue( simulator.getActiveEvents().size() == 2 );
		assertTrue( simulator.getActiveStates().get( 0 ).getName().equals( "CD-Wechsler-Betrieb" ) );		

		System.out.println( simulator.getActiveEvents() );

		// As CD_manuell comes before Radio_manuell, no event removed from queue
		// and both events are swaped
		simulator.doStep();
		// Now, the needed event (Radio_manuell) is on top of queue
		simulator.doStep();
		assertTrue( simulator.getActiveEvents().size() == 1 );
		assertTrue( simulator.getActiveStates().get( 0 ).getName().equals( "Radiobetrieb" ) );		

		simulator.doStep();
		assertTrue( simulator.getActiveEvents().size() == 0 );
		assertTrue( simulator.getActiveStates().get( 0 ).getName().equals( "CD-Wechsler-Betrieb" ) );		
		
	}

	@Ignore
	public void testSendSignal() {
		fail("No signals in model!");
	}

	@Test
	public void testDoStep() throws NoValidStateMachineException, ExpressionParseException, ExpressionEvaluationException {
		ArrayList<MyEvent> events = simSM.getAvailableEvents();
		
		MyEvent call_CD_eingelegt = null;
		MyEvent call_Radio_manuell = null;
		MyEvent call_CD_manuell = null;
		for ( int i = 0; i < events.size(); i++ ) {
			MyEvent myEvent = events.get( i );
			if ( myEvent.getEventType() == EventType.callOperation ) {
				Operation operation = myEvent.getOperation();
				if ( operation.getName().equalsIgnoreCase( "CD_eingelegt" ) ) {
					call_CD_eingelegt = simulator.callOperation( operation );
				}
				if ( operation.getName().equalsIgnoreCase( "Radio_manuell" ) ) {
					call_Radio_manuell = simulator.callOperation( operation );
				}
				if ( operation.getName().equalsIgnoreCase( "CD_manuell" ) ) {
					call_CD_manuell = simulator.callOperation( operation );
				}
			}
		}
		
		Parameter parameterB = call_CD_eingelegt.getOperation().getOwnedParameter( "b", null );
		simulator.setParameterValue( call_CD_eingelegt, parameterB, "true" );

		assertTrue( simulator.getActiveEvents().size() == 3 );

		simulator.doStep();
		assertTrue( simulator.getActiveStates().get( 0 ).getName().equals( "Radiobetrieb" ) );		

		simulator.doStep();
		assertTrue( simulator.getActiveStates().get( 0 ).getName().equals( "CD-Wechsler-Betrieb" ) );		

		simulator.doStep();
		assertTrue( simulator.getActiveStates().get( 0 ).getName().equals( "Radiobetrieb" ) );			
	}

	@Test
	public void testGetActiveEvents() {
		assertTrue( simulator.getActiveEvents().size() == 0 );
	}

	@Test
	public void testGetVariableValues() throws ExpressionEvaluationException {

		Object value = simulator.getVariable( "CD_drin" );
		if ( value instanceof Boolean ) {
			Boolean bool = ( Boolean ) value;
			assertTrue( bool.booleanValue() );
		} else {
			fail( "Invalid type of CD_drin!" );
		}		
		
		value = simulator.getVariable( "Kassette_drin" );
		if ( value instanceof Boolean ) {
			Boolean bool = ( Boolean ) value;
			assertFalse( bool.booleanValue() );
		} else {
			fail( "Invalid type of Kasette_drin!" );
		}		
		
	}

	@Test
	public void testGetVariableTypes() throws ExpressionEvaluationException {
		
		java.lang.Class class1 = simulator.getVariableType( "CD_drin" );
		java.lang.Class class2 = simulator.getVariableType( "Kassette_drin" );
		
		assertTrue( class1 == Boolean.class && class2 == Boolean.class );	
	}

	@Test
	public void testGetVisitedVertices() {
		assertTrue( simulator.getVisitedVertices().size() == 11 );
	}

	@Test
	public void testGetPassedTransitions() {
		assertTrue( simulator.getPassedTransitions().size() == 9 );
	}

	@Test
	public void testSimulate() throws NoValidStateMachineException, ExpressionEvaluationException, ExpressionParseException {
		
		// Rest simualation
		simulator.setupSimulation();
		
		ArrayList<MyEvent> events = simSM.getAvailableEvents();
		MyEvent call_CD_eingelegt = null;
		MyEvent call_Radio_manuell = null;
		MyEvent call_CD_manuell = null;
		MyEvent call_ausschalten = null;
		for ( int i = 0; i < events.size(); i++ ) {
			MyEvent myEvent = events.get( i );
			if ( myEvent.getEventType() == EventType.callOperation ) {
				Operation operation = myEvent.getOperation();
				if ( operation.getName().equalsIgnoreCase( "CD_eingelegt" ) ) {
					call_CD_eingelegt = simulator.callOperation( operation );
				}
				if ( operation.getName().equalsIgnoreCase( "Radio_manuell" ) ) {
					call_Radio_manuell = simulator.callOperation( operation );
				}
				if ( operation.getName().equalsIgnoreCase( "CD_manuell" ) ) {
					call_CD_manuell = simulator.callOperation( operation );
				}
				if ( operation.getName().equalsIgnoreCase( "ausschalten" ) ) {
					call_ausschalten = simulator.callOperation( operation );
				}
			}
		}
		
		assertTrue( simulator.simulate() == StopReason.totalDeadlock );
		assertTrue( simulator.getActiveEvents().size() == 2 );
		assertTrue( simulator.simulate() == StopReason.totalDeadlock );
		// CD_manuell remains in queue as CD_drin is false
		assertTrue( simulator.getActiveEvents().size() == 1 );
		// ==> add cd_eingelegt with true
		simulator.addEvent( call_CD_eingelegt );
		Parameter parameterB = call_CD_eingelegt.getOperation().getOwnedParameter( "b", null );
		simulator.setParameterValue( call_CD_eingelegt, parameterB, "true" );	
		// ==> add Radio_manuell
		simulator.addEvent( call_Radio_manuell );
		// Simulator until queue is empty
		simulator.setStopIfTotalDeadlock( false );
		simulator.setStopIfNoMoreEvents( true );
		
		assertTrue( simulator.simulate() == StopReason.NoMoreEvents );	
		assertTrue( simulator.getActiveStates().size() == 1 );	
		assertTrue( simulator.getActiveStates().get(0).getName().equals( "CD-Wechsler-Betrieb" ) );
	}

}
