package de.fau.cs.swe.da.modelsimulator;

import org.eclipse.emf.common.util.*;
import org.eclipse.uml2.uml.*;
import java.util.*;
import de.cnc.expression.exceptions.*;
import de.fau.cs.swe.da.exceptions.NoValidStateMachineException;
import de.fau.cs.swe.da.modelsimulator.MyEvent.EventType;

import java.lang.Class;
import java.math.BigDecimal;

/**
 * 
 * Main simulation class.
 * 
 * @author Dominik Schindler
 * 
 * 
 * Version history:
 * 2007-01-31
 * 		- checkTrigger() complety rewritten, as the old behavior was not correct
 * 
 * 2007-01-25
 * 		- some bugfixes
 * 		- JUnit text cases
 * 
 * 2007-01-22
 * 		- added pseudo parallel multiple concurrent entry/exit behavior
 * 		- some bug fixes concering the history states
 *
 */
public class Simulator implements Observer {
	// Stores *all* active states in the statemachine
	private ArrayList<Vertex> activeStates = new ArrayList<Vertex>();
	// Used to store all states, which are active next round
	private ArrayList<Vertex> nextActiveStates = new ArrayList<Vertex>();
	// Declared, but currently not really used
	private ArrayList<Region> activeRegions = new ArrayList<Region>();
	// Used to store the states when exiting a complex state 
	private Hashtable<Region,ArrayList<Vertex>> historyStates = new Hashtable<Region,ArrayList<Vertex>>(); 
	// Used to store the triggers for a complex state
	private Hashtable<State,EList> externalTransitions = new Hashtable<State,EList>();
	// Used to map value to parameter when an opartion is called
	private Hashtable<MyEvent, Hashtable<Parameter, String>> parameterConfigurations = new Hashtable<MyEvent, Hashtable<Parameter,String>>();
	// Holds the current simulateable statemachine
	private AbstractSimulateableStateMachine assm;
	// hHolds the state machine from the object above
	private StateMachine sm;
	// Round counter, increased once by a round
	private int round = 0;
	// Used to store the variables, types and values for the expression parser 
	private MyRuntimeEnvironment runtimeEnvironment = null;
	// Holds all transitions and vertices passed (ordered)
	private ArrayList<Transition> passedTransitions = new ArrayList<Transition>();
	private ArrayList<Vertex> visitedVertices = new ArrayList<Vertex>();
	// The events "queue"
	private ArrayList<MyEvent> activeEvents = new ArrayList<MyEvent>();
	// When stop automated simulation
	private boolean stopIfNoMoreEvents = false;
	private boolean partialDeadlock = true; // not yet implemented
	private boolean stopIfTotalDeadlock = true;
	
	/**
	 * What to do with triggers on toop of the queue and currently not needed.
	 */
	public enum EventsQueueBehavior { 
		/**
		 * Discard event on top of queue
		 */
		discard, 
		/**
		 * Readd event on top of queue at the and.
		 */
		reAdd, 
		/**
		 * Search for the event in queue. 
		 */
		skip 
	};
	private EventsQueueBehavior eventsQueueBehavior = EventsQueueBehavior.reAdd;

	/**
	 * In which order should multiple concurrent entry/exit behaviors be 
	 * executed.
	 */
	public enum MultipleConcurrentBehavior { 
		/**
		 * Do all entry/exit behaviors of a complete path sequentially.
		 */
		sequential, 
		/**
		 * Do the entry/exit behavior pseudo parallel.
		 */
		parallel 
	};
	private MultipleConcurrentBehavior multipleConcurrentBehavior = MultipleConcurrentBehavior.sequential;

	/**
	 * Why the autoamtic simulation has been stopped.
	 */
	public enum StopReason { 
		/**
		 *  
		 */
		None, 
		/**
		 * No more events in the active events queue.
		 */
		NoMoreEvents, 
		/**
		 * Partial deadlock occured (not implemented)
		 */
		partialDeadlock, 
		/**
		 * Total deadlock occured. 
		 */
		totalDeadlock, 
		/**
		 * Terminiator state reached.
		 */
		terminated 
	};
	private StopReason stopReason = StopReason.None;  
	// Special flag: If a transition was passed and the current state for a round 
	// is no more active, stop handling of the remaining outgoings transtions or not 
	// true = UML specification
	private boolean breakOutgoingForLoop = true; 
	// For debuging purpose
	private boolean showOutput = true;
	//
	private MyEvent thisRoundActiveEvent = null; 
	// Is true, if no trigger needed ==> do not remove event from queue
	private boolean noTrigger = true;
	
	
/*
 * ====================== PUBLIC PART ======================
 */
	
	/**
	 * 
	 * Constructor. 
	 * 
	 * @param ssm - The simulateable state machine to be simulated.
	 */
	public Simulator( AbstractSimulateableStateMachine ssm ) {
		this.assm = ssm;
		this.sm = ssm.getStateMachine();
		// use runtime environement if on has been created
		MyRuntimeEnvironment runEnv = ssm.initalRuntimeEnvironement();
		runtimeEnvironment = ( runEnv != null ) ? runEnv : new MyRuntimeEnvironment();
		runtimeEnvironment.addObserver( this );
	}
	
	
	/**
	 * 
	 * Sets basic variables in the runtime environement.
	 *
	 */
	private void setupBasicVariables() {
		try {
			runtimeEnvironment.setVariableType( "round", BigDecimal.class );
			runtimeEnvironment.setVariable( "round", new BigDecimal( 0 ) );
		} catch ( Exception ex ) {
			
		}
	}
	
	/**
	 * 
	 * Initializes all lists and sets the overall inital pseudostate as current
	 * active state, which belongs to the statemachine itself.
	 * 
	 * @throws NoValidStateMachineException
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 */
	public void setupSimulation() throws NoValidStateMachineException, ExpressionEvaluationException, ExpressionParseException {
		setupBasicVariables();		
		if ( sm != null && sm.getRegions().size() != 1 )
			throw new NoValidStateMachineException( "None or more than one region found in state machine!" );
		// if no region specified, select first region (=sm) as start region
		Region region = ( Region ) sm.getRegions().get( 0 );
		// clear all variant lists
		activeStates.clear();
		activeRegions.clear();
		activeEvents.clear();
		visitedVertices.clear();
		passedTransitions.clear();
		stopReason = StopReason.None;
		historyStates.clear();
		externalTransitions.clear();
		parameterConfigurations.clear();
		// do setup
		setupSimulation( region );
		this.round = 0;
	}
	
	/**
	 * 
	 * Searches for a initial state in the region and sets is as current active 
	 * state. 
	 * 
	 * @param region - The region where to search and set the inital state.
	 * @throws NoValidStateMachineException - Thrown, if the region has no initial state.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * 
	 */
	public void setupSimulation( Region region ) throws NoValidStateMachineException, ExpressionEvaluationException, ExpressionParseException {		
		addActiveRegion( region );

		// search for overall initial pseudostate in region
		for ( int i = 0; i < region.getSubvertices().size(); i++ ) {
			Object vertex = region.getSubvertices().get( i );
			if ( vertex instanceof Pseudostate ) {
				Pseudostate ps = ( Pseudostate ) vertex;
				if ( ps.getKind() == PseudostateKind.INITIAL_LITERAL ) {
					addActiveState( ps );
					return;
				}
			}
		}
		throw new NoValidStateMachineException( "No overall initial state found!" );
	}		
	
	/**
	 * 
	 * Sets the variable "name" to "value" in the runtime environement, which
	 * is used to evaluate expressions.
	 * 
	 * <b>Note:</b> Unknown variables in expressions throw an exception, except 
	 * assign expressions, where the variable will be added automatically. 
	 * 
	 * @param name - Name of the variable 
	 * @param value - The value of the variable. Musst fit to the specified type.
	 * @throws ExpressionEvaluationException - Thrown, if the variable is a reserved word or the value is not valid.
	 */
	public void setVariable( String name, Object value ) throws ExpressionEvaluationException {
		runtimeEnvironment.setVariable( name, value );
	}
	
	/**
	 * 
	 * Sets the behavior of the active event queue.
	 * 
	 * @param eqb - A value of the EventsQueueBehavior enumeration.
	 */
	public void setEventQueueBehavior ( EventsQueueBehavior eqb ) {
		this.eventsQueueBehavior = eqb;
	}
	
	/**
	 * 
	 * Returns the current beavior of the event queue.
	 * 
	 * @return A value of the EventsQueueBehavior enumeration.
	 */
	public EventsQueueBehavior getEventQueueBehavior() {
		return this.eventsQueueBehavior;
	}
	
	/**
	 * 
	 * Sets the behavior if multiple concurrent entry/exit behaviors occure.
	 * 
	 * @param mcq - A value of the MultipleConcurrentBehavior enumeration.
	 */
	public void setMultipleConcurrentBehavior( MultipleConcurrentBehavior mcq ) {
		this.multipleConcurrentBehavior = mcq;
	}
	
	/**
	 * 
	 * Returns the current set behavior if multiple concurrent entry/exit 
	 * behaviors occure.
	 * 
	 * @return A value of the MultipleConcurrentBehavior enumeration.
	 */
	public MultipleConcurrentBehavior getMultipleConcurrentBehavior() {
		return this.multipleConcurrentBehavior;
	}
	
	/**
	 * 
	 * Removes the variable from the runtime environement.
	 * 
	 * @param name - The variable name to be removed.
	 * @throws ExpressionEvaluationException 
	 */
	public void removeVariable( String name ) throws ExpressionEvaluationException {
		runtimeEnvironment.removeVariable( name );
	}
	
	/** 
	 * 
	 * Adds an event to the event queue.
	 * 
	 * <b>Important:</b> If you wannt to add a operation with paramters to the
	 * active events list, you <b>must</b> use the call operation method, which 
	 * creates a new object with the associated operation, or clone a MyEvent 
	 * object with the copy constructor, as the MyEvent object is used as key of
	 * the parameteres hashtable, wich stores the values to a parameter. If you do
	 * not clone the object, you'll will have a problem if you want to add the 
	 * MyEvent object with the same operation/signal twice, as the second object
	 * would affect the value of the first object as the hash code is the same.
	 * 
	 * @param event - A MyEvent object to be added ad the end of the queue.
	 */
	public void addEvent( MyEvent event ) {
		activeEvents.add( event );
	}
	
	/**
	 * 
	 * Adds a call to the end of the event queue and returns the event. Dont't 
	 * forget to call the setCallEventParameter() method if the operation has 
	 * parameteres to be set.
	 * 
	 * @param operation - A refernce to an operation, which is called.
	 * @return The MyEvent object, which represents the call of the operation.
	 */
	public MyEvent callOperation( Operation operation ) {
		MyEvent myEvent = new MyEvent( operation );
		activeEvents.add ( myEvent );
		return myEvent;
	}
	
	/**
	 * 
	 * Adds a signal sent to the end of the event queue. 
	 * 
	 * @param signal - A reference to a signal to be sent.
	 * @return The MyEvent object, which represents the signal sent.
	 */
	public MyEvent sendSignal( Signal signal ) {
		MyEvent myEvent = new MyEvent( signal );
		activeEvents.add ( myEvent );
		return myEvent;		
	}
	
	/**
	 * 
	 * Adds a complete event list at the and of the event queue.
	 * 
	 * @param al - A ArrayList of MyEvent objects. 
	 */
	
	public void addEvents( ArrayList<MyEvent> al ) {
		activeEvents.addAll( al );
	}
	
	/**
	 * 
	 * Removes an event from the active event list by event.
	 * 
	 * @param event - The event to be removed.
	 */
	public void removeActiveEvent( MyEvent event ) {
		activeEvents.remove( event );
	}
	
	/**
	 * 
	 * Removes an event from the event list by index.
	 * 
	 * @param index - The index of the event to be removed.
	 */
	public void removeActiveEvent( int index ) {
		activeEvents.remove( index );
	}
	
	/**
	 * 
	 * Stop automated simulation if no more events in the event queue.
	 * 
	 * @param b - Yes (true) or no (false)
	 */
	public void setStopIfNoMoreEvents( boolean b ) {
		stopIfNoMoreEvents = b;
	}
	
	/**
	 * 
	 * Returns, if the automated simulation is stopped if the events queue is empty. 
	 * 
	 * @return True (yes) or false (no)
	 */
	public boolean getStopIfNoMoreEvents() {
		return stopIfNoMoreEvents;
	}
	
	/**
	 * 
	 * Stop automated simulation if partial deadlock occured (not yet implemented).
	 * 
	 * @param b - Yes (true) or no (false)
	 */
	public void setStopIfPartialDeadlock( boolean b ) {
		partialDeadlock = b;
	}
	
	/**
	 * 
	 * Returns, if the automated simulation is stopped if partial deadlock 
	 * occured (not yet implemented).
	 * 
	 * @return True (yes) or false (no)
	 */
	public boolean getStopIfPartialDeadlock() {
		return partialDeadlock;
	}
	
	/**
	 * 
	 * Stop automated simulation if total deadlock occured.
	 * 
	 * @param b - Yes (true) or no (false)
	 */
	public void setStopIfTotalDeadlock( boolean b ) {
		stopIfTotalDeadlock = b;
	}
	
	/**
	 * 
	 * Returns, if the automated simulation is stopped if total deadlock 
	 * occured.
	 * 
	 * @return True (yes) or false (no)
	 */
	public boolean getStopIfTotalDeadlock() {
		return stopIfTotalDeadlock;
	}
	
	/**
	 * 
	 * Sets the myEvents list as the active events queue.
	 * 
	 * @param myEvents - The ArrayList of MyEvent objects to be set.
	 */
	
	public void setActiveEvents(ArrayList<MyEvent> myEvents) {
		activeEvents.clear();
		activeEvents.addAll( myEvents );
	}
	
	/**
	 * 
	 * Used to set the result of a subsimulated statemachine (e.g. with
	 * doStateMachine()).
	 * 
	 * @param stopReason - A value of the StopReason enumeration to be set.
	 */
	public void setStopReason( StopReason stopReason ) {
		this.stopReason = stopReason;
	}
	
	/**
	 * 
	 * Main simulation method!
	 * 
	 * Does a round in the simulation by searching for events, which match 
	 * the triggers of the outgoing transitions of the active state. Guards are
	 * also checked. 
	 * 
	 * If the head of the event list matches a trigger, the event is performed
	 * and the event is removed from the queue. If no trigger present, only
	 * the guard will be checked an the effect is fired.  
	 *
	 * @return A value of the StopReason enumeration. This value is used by the automated simualtion method simulate(). 
	 * @throws NoValidStateMachineException - Thrown, if the statmachine is not valid.
	 * @throws ExpressionEvaluationException - Thrown, if the evaluation of an expression was faulty.
	 * @throws ExpressionParseException - Thrown, if an expression is not valid.
	 *
	 */
	public StopReason doStep() throws NoValidStateMachineException, ExpressionParseException, ExpressionEvaluationException {
		
		int oldTransitionsCount = passedTransitions.size();
		if ( showOutput ) System.out.println( "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ STEP ( " + round + " ) ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~" );
		
		// Get 1st event wich will be active for this round
		if ( activeEvents.size() > 0 ) {
			thisRoundActiveEvent = activeEvents.remove( 0 );
		} else {
			thisRoundActiveEvent = null;
		}
	
		noTrigger = true;		
		
		if ( showOutput ) System.out.println( "THIS ROUND ACTIVE: " + thisRoundActiveEvent );

		// Do a step for all states in the active states list
		while ( activeStates.size() > 0 ) {
			Vertex state = activeStates.get( 0 );
			activeStates.remove( state );
			if ( showOutput ) System.out.println( "Current state: " + state );
			// Check for the state, if an outgoing transition can be passed.
			doSingleStep( state );
		}
		
		if ( showOutput ) System.out.println( "END OF ROUND: " + thisRoundActiveEvent );
		
		// True, if no trigger in this round 
		if ( !noTrigger ) {
			// If event not consumed
			if ( thisRoundActiveEvent != null ) {
				// If reAdd slected, add unused event to end of queue
				if ( eventsQueueBehavior == EventsQueueBehavior.reAdd ) {
					activeEvents.add( thisRoundActiveEvent );
				}
				// Else remove event from top of queue
			} else {
				// Event consumed ( event == null ) ==> remove event from top of queue
			}
		} else {
			if ( thisRoundActiveEvent != null ) 
				activeEvents.add( 0, thisRoundActiveEvent );
		}
		
		// If no transition has been added to the passedTransitions queue, 
		// and the flag ist set, stop the simualtion an return the reason.
		if ( passedTransitions.size() == oldTransitionsCount &&
				stopIfTotalDeadlock ) {
			stopReason = StopReason.totalDeadlock;
		}

		// Add all active states for the next round to the active states queue.
		activeStates.addAll( nextActiveStates );
		nextActiveStates.clear();

		if ( showOutput ) System.out.println( "HistoryStates: " + historyStates );
		
		// Remove unactive events from the activeEvents queue.
//		activeEvents.removeAll( nextUnActiveEvents );
//		nextUnActiveEvents.clear();			
		
		// Increase round variable and add it to the runtime environment
		round++;
		try {
			de.cnc.expression.Expression expparser = de.cnc.expression.Expression.parse( "round := " + round , runtimeEnvironment );
			expparser.eval( runtimeEnvironment );
		} catch ( Exception ex ) {}				
		return stopReason;
	}
	
	private void doSingleStep( Vertex currentVertex ) throws NoValidStateMachineException, ExpressionParseException, ExpressionEvaluationException {
		
		// Check, if there are events in the events queue.
		if ( activeEvents.size() == 0 && stopIfNoMoreEvents ) {
			stopReason = StopReason.NoMoreEvents;
		}
				
/* === Handling of history and terminator state === */				
		
		// Check, if current vertex is a history state.
		if ( currentVertex instanceof Pseudostate ) {
			Pseudostate ps = ( Pseudostate ) currentVertex;
			// If the current vertex is a history state, check the historyStates
			// hasttable for last active vertices in the region.
			if ( ps.getKind() == PseudostateKind.SHALLOW_HISTORY_LITERAL ) {
				Region region = ps.getContainer();
				// Normaly not needed, but how knows?
				if ( region != null ) {
					ArrayList<Vertex> activeElements = historyStates.get( region );
					if ( activeElements != null  ) {
						if ( activeElements.size() == 1 && activeElements.get(0) == currentVertex ) {
							
						} else {
							addNextActiveStatesWithoutBehavior( activeElements );
							return;
						}
					} else {
						// First entry, do nothing ;-)
					}
				}
			}
			
			if ( ps.getKind() == PseudostateKind.DEEP_HISTORY_LITERAL ) {
				Region region = ps.getContainer();
				// Not needed normaly, but how knows?
				if ( region != null ) {
					setDeepHistoryStatesRec( region );
					return;
				}
			}	
			
			// Terminator state => abort simulation and set flag.
			if ( ps.getKind() == PseudostateKind.TERMINATE_LITERAL ) {
				activeStates.clear();
				stopReason = StopReason.terminated;
				return;
			}
		}
		
/* === Handling of the final states in complex states === */		
		
		// Check, if final state is in a region of a complex state
		if ( currentVertex instanceof FinalState ) {
			if ( showOutput ) System.out.println( "FinalState: " + currentVertex );
			FinalState finalState = ( FinalState ) currentVertex;
			State state = finalState.getContainer().getState();
			// If final state is in a complex state ...
			if ( state != null ) {
				EList regions = state.getRegions();
				//  ... check, if it is a composite or an orthogonal state.				
				if ( regions.size() > 1 ) {
					ArrayList<Vertex> finalStates = new ArrayList<Vertex>(); 
					ArrayList<Vertex> activeFinalStates = new ArrayList<Vertex>();
					for ( int i = 0; i < regions.size(); i++ ) {
						Region region = ( Region ) regions.get( i );
						if ( showOutput ) System.out.println( "Region: " + region );
						EList elements = region.getOwnedElements();
						// Collect all final states in other regions
						for ( int j = 0; j < elements.size(); j++ ) {
							Object element = elements.get( j );
							if ( element instanceof FinalState && element != finalState ) {
								finalStates.add( ( FinalState ) element );
								if ( showOutput ) System.out.println( "FinalState added: " + element );
							}
						}
						// Collect all other active final states of the regions
						for ( int j = 0; j < activeStates.size(); j++ ) {
							Vertex vertex = activeStates.get( j );
							if ( vertex instanceof FinalState && vertex != finalState
									&& vertex.getContainer() == region ) {
								activeFinalStates.add( ( FinalState ) vertex );
							}
						}				
					}
										
					if ( showOutput ) System.out.println( "FinalStates: " + finalStates );
					if ( showOutput ) System.out.println( "--> Active: " + activeFinalStates );
						
					// Check, if all final states of an orthogonal region are active
					if ( activeFinalStates.size() > 0 &&
							finalStates.containsAll( activeFinalStates ) ) {
						activeStates.removeAll( finalStates );
						// visitedVertices.addAll( finalStates );
						// visitedVertices.add( finalState );
						nextActiveStates.add( state );
						if ( showOutput ) System.out.println("DEFAULT ORTHOGONAL EXIT!");
						externalTransitions.remove( state );
					return;
					}
				} else {
					nextActiveStates.add( state );
					if ( showOutput ) System.out.println("DEFAULT EXIT!");
					externalTransitions.remove( state );
					return;
				}
			} 
		}
		
		// Used to determine, if a transition except "else" has been passed.
		boolean atLeastOneTransitionPassed = false;

		// First of all, check if trigger of a complex state is trigged.
		// If any transitions are in the list, a complex state is active		
		if ( showOutput ) System.out.println( "CHECK EXTERNAL HASHTABLE: " + externalTransitions );

		java.util.Enumeration en = externalTransitions.keys();
		while ( en.hasMoreElements() ) {
			Object key = en.nextElement();
			EList transitions = externalTransitions.get( key );
			for ( int i = 0; i < transitions.size(); i++ ) {
				Transition transition = ( Transition ) transitions.get( i );
				EList triggers = transition.getTriggers();
				for ( int j = 0; j < triggers.size(); j++ ) {
					Trigger trigger = ( Trigger ) triggers.get( j );
					
					Event event = trigger.getEvent();
					MyEvent myEvent = null;
					// Check, if trigger event is in the event list.
					if ( ( myEvent = checkTrigger ( event ) ) != null ) {
						// If trigger fired, check guard.
						if ( checkGuard( transition, myEvent ) ) {
							if ( showOutput ) System.out.println( "--> EXTERNAL EVENT: " + event );
							State state = ( State ) key;
							addNextActiveState( transition.getTarget(), state, transition );
							atLeastOneTransitionPassed = true;
							// If associated event is a operation call, do operation
							if ( thisRoundActiveEvent.getEventType() == EventType.callOperation ) {
								doOperation( thisRoundActiveEvent, thisRoundActiveEvent.getOperation() );
							}
							// Remove event at end of round
							thisRoundActiveEvent = null;
						}	
					}
				}
			}

			if ( atLeastOneTransitionPassed ) {
				// Remove all active substates from the complex state.
				if ( showOutput ) System.out.println( "--> EXTERNAL TRANSITION TRIGGERED:" );				
				State state = ( State ) key;
				EList regions = state.getRegions();
				for ( int j = 0; j < regions.size(); j++ ) {
					Region region = ( Region ) regions.get( j );
					EList elements = region.getOwnedElements();
					for ( int k = 0; k < elements.size(); k++ ) {
						Object object = elements.get( k );
						if ( activeStates.contains( object ) ) {
							activeStates.remove( object );
						}
						if ( nextActiveStates.contains( object ) ) {
							nextActiveStates.remove( object );
						}
					}
				}
				return;
			}
		}
		
		EList outgoings = currentVertex.getOutgoings();
		Transition elseTransition = null;
	
/* === Handling of default exit === */		
		
		// If current vertex is a complex state ==> default exit ==>
		// pass only untriggered transition if a trigger is also active!		
		if ( currentVertex instanceof State ) {
			State state = ( State ) currentVertex;
			if ( state.isOrthogonal() || state.isComposite() || state.getRegions().size() > 0 ) {
				for ( int i = 0; i < outgoings.size(); i++ ) {
					Transition transition = ( Transition ) outgoings.get( i );
					if ( transition.getGuard() != null || ( transition.getTriggers() != null &&
							transition.getTriggers().size() > 0 ) ) {
						continue;
					}
					if ( showOutput ) System.out.println( "UNTRIGGERED");
					addNextActiveState( transition.getTarget(), currentVertex, transition );
					return;
				}
			}
		}

/* === Handling of forks === */		

		if ( currentVertex instanceof Pseudostate ) {
			Pseudostate ps = ( Pseudostate ) currentVertex;
			// Check, if pseudostate is a fork and if outgoing transitions are
			// not in the same region ==> explicit entry of an orthogonal region.	
			if ( ps.getKind() == PseudostateKind.FORK_LITERAL ) { 
				
				if ( multipleConcurrentBehavior == MultipleConcurrentBehavior.sequential ) {
					for ( int i = 0; i < outgoings.size(); i++ ) {
						Transition transition = ( Transition ) outgoings.get( i ); 
						Vertex target = transition.getTarget();
						addNextActiveState( target, currentVertex, transition );
					}				
				} else {

					if (  multipleConcurrentBehavior == MultipleConcurrentBehavior.parallel ) {
						ArrayList<ArrayList<Region>> rls = new ArrayList<ArrayList<Region>>();
						ArrayList<Region> rl1 = getPathToRoot( currentVertex.getContainer() );
						for ( int i = 0; i < outgoings.size(); i++ ) {
							Transition transition = ( Transition ) outgoings.get( i );
							ArrayList<Region> rl2 = getPathToRoot( transition.getTarget().getContainer() );
							Region common = getCommonRegion( rl1 , rl2 );
							rls.add( invertRegionsList( getAllRegionsUntil( rl2, common ) ) );
							Vertex target = transition.getTarget();							
							//addNextActiveState( target, currentVertex, transition );
							passedTransitions.add( transition );
							transitionPassed( transition );
							// If a complex state is entered, collect the outgoing transitions and
							// if a complex state is left, remove outgoing transitions of the 
							// outgoingsTrantions hashtable
							collectOutgoingTransitionsRec( target, currentVertex );
							nextActiveStates.add( target );					
						}
						
						if ( showOutput ) System.out.println( "COMMON: " + rls );
						
						Region sourceRegion = currentVertex.getContainer();

						int max = getMaxRegionsInList( rls );
						int count = rls.size() * max;
						// Go throug all lists an pick the region on top of
						// each list.
						for ( int i = 0; i < count; i++ ) {
							ArrayList<Region> temp = rls.get( i % rls.size() );
							Region region = popRegionsArrayList( temp );
							if ( region != null ) {
								if ( sourceRegion != region ) {
									// Store active region (not used yet).
									activeRegions.remove( sourceRegion );
									activeRegions.add( region );
								}
								State state = region.getState();
								if ( state != null ) {
									doEntry( state );
									visitedVertices.add( state );
								}
							}
						}
						
						// Finally, do the entry behavior of the target states
						// and add them to the visitedVertices list.
						for ( int i = 0; i < outgoings.size(); i++ ) {
							Transition transition = ( Transition ) outgoings.get( i );
							Vertex target = transition.getTarget();
							if ( target instanceof State ) {
								doEntry( ( State ) target );
							}
							visitedVertices.add( target );
						}
					}
				}
				return;
			}
		}
		
/* === Handling of a choice state with else guard === */	
		
		// Search for optional else transition
		for ( int i = 0; i < outgoings.size(); i++ ) {
			Transition transition = ( Transition ) outgoings.get( i );
			Constraint guard = transition.getGuard();
			if ( guard != null ) {
				ValueSpecification specification = guard.getSpecification();
				if ( specification instanceof OpaqueExpression ) {
					OpaqueExpression expression = ( OpaqueExpression ) specification;
					for ( int j = 0; j < expression.getBodies().size(); j++ ) {
						String exp = ( String ) expression.getBodies().get( j );
						String lang = ( String ) expression.getLanguages().get( j );
						if ( !lang.equals( "Expression" ) ) continue;
						if ( exp.equalsIgnoreCase( "else" ) ) {
							elseTransition = transition; 
							if ( showOutput ) System.out.println( "Else: " + elseTransition );
						}
					}
				}
			}
		}
		
/* === Handling of "normal" states === */	
		
		// Go through all outgoing transitions an find one, with trigger in event 
		// list and true guard.
		for ( int i = 0; i < outgoings.size(); i++ ) {
			Transition transition = ( Transition ) outgoings.get( i ); 
			EList triggers = transition.getTriggers();
			if ( showOutput ) System.out.println( "Outgoing: " + ( ( transition.getName() != null ) ? transition.getName() : transition ) );
			if ( showOutput ) System.out.println( "Triggers: " + triggers );
			if ( showOutput ) System.out.println( "getTarget(): " + ( ( transition.getTarget().getName() != null ) ? transition.getTarget().getName() :  transition.getTarget() ) );							

			Vertex target = transition.getTarget();
			// If target already in the list, do nothing.
			if ( activeStates.contains( target ) || ( nextActiveStates.contains( target ) ) ) {
				continue;
			}			
			
			if ( triggers.size() == 0 ) {
				// If no trigger defined, check guard only.
				if ( checkGuard( transition, null ) ) {
					if ( showOutput ) System.out.println( "NO TRIGGER");
					addNextActiveState( transition.getTarget(), currentVertex, transition );
					atLeastOneTransitionPassed = true;
					//thisRoundActiveEvent = null;
				}
			} else {
				// Go through outgoing transitions and find one with trigger = active 
				// event.
				
				noTrigger = false;

				for ( int j = 0; j < triggers.size(); j++ ) {
					Trigger trigger = ( Trigger ) triggers.get( j );
					Event event = trigger.getEvent();
					if ( showOutput ) System.out.println( "Needed Trigger: " + ( ( event.getName() != null ) ? event.getName() : event ) );
					// Check, if trigger is in the event list
					MyEvent myEvent = null;
					if ( ( myEvent = checkTrigger( event ) ) != null ) {						
						// If trigger fired, check guard.
						if ( checkGuard( transition, myEvent ) ) {
							State complexState = transition.getSource().getContainer().getState();
							// Check, if sourceState is inside a complex state. 
							if ( complexState != null && complexState instanceof State ) {
								Region sourceRegion = transition.getSource().getContainer();
								Region targetRegion = transition.getTarget().getContainer();
								// Check, if transition points to a vertex outside the current region.
								// If yes, the complex stat ist left and the states inside are removed.
								if ( sourceRegion != targetRegion ) {
									EList regions = complexState.getRegions();
									for ( int k = 0; k < regions.size(); k++ ) {
										Region region = ( Region ) regions.get( k );
										EList elements = region.getOwnedElements();
										for ( int l = 0; l < elements.size(); l++ ) {
											Object object = elements.get( l );
											if ( activeStates.contains( object ) ) {
												if ( showOutput ) System.out.println( "Removed " + object );
												activeStates.remove( object );
											}
											if ( nextActiveStates.contains( object ) ) {
												if ( showOutput ) System.out.println( "Next active removed " + object );
												nextActiveStates.remove( object );
											}
										}
									}
								}
							}
							
							addNextActiveState( transition.getTarget(), currentVertex, transition );
							atLeastOneTransitionPassed = true;
							// If associated event is a operation call, do operation
							if ( thisRoundActiveEvent.getEventType() == EventType.callOperation ) {
								doOperation( thisRoundActiveEvent, thisRoundActiveEvent.getOperation() );
							}
							thisRoundActiveEvent = null;
							break;
						} else {
//							if ( eventsQueueBehavior == EventsQueueBehavior.reAdd ) {
//								activeEvents.add( myEvent );
//							} else {
//								nextUnActiveEvents.add( myEvent );
//							}
//							System.out.println( nextUnActiveEvents );
						}
					}
				}
			}
			
			// If one outgoing transtion is passed, the state ist left and the other
			// outgoing transitions are not checked.
			if ( atLeastOneTransitionPassed && breakOutgoingForLoop ) {
				break;
			}
		}

		// If guards of all outgoing transitions are false, pass the else transition.
		if ( ( outgoings.size() != 0 ) && !atLeastOneTransitionPassed ) {
			if ( elseTransition != null ) {
				addNextActiveState( elseTransition.getTarget(), currentVertex, elseTransition );
				atLeastOneTransitionPassed = true;
			}
		}			
		
		// If no transition is passed, add old state to active states list. 
		if ( !atLeastOneTransitionPassed ) {
			addNextActiveStateWithoutBehavior( currentVertex );
		}
		
	}
	
	/**
	 * 
	 * Returns the event queue.
	 * 
	 * @return The ArayList with active MyEvent objects.
	 */
	public ArrayList<MyEvent> getActiveEvents() {
		return activeEvents;
	}
		
	/**
	 * 
	 * Returns the value of the variable name as a wrapper object. 
	 * 
	 * @param name - The variable name.
	 * @return The wrapper object with the value.
	 * @throws ExpressionEvaluationException
	 */
	public Object getVariable ( String name ) throws ExpressionEvaluationException {
		return runtimeEnvironment.getVariable( name );
	}

	/**
	 * 
	 * Returns the type of the variable name as a class object. 
	 * 
	 * @param name - The variable name.
	 * @return The class object.
	 * @throws ExpressionEvaluationException
	 */
	public Class getVariableType ( String name ) throws ExpressionEvaluationException {
		return runtimeEnvironment.getVariableType( name );
	}
	
	/**
	 * 
	 * Returns all defined variable values in the runtime environment.
	 * 
	 * @return All defined variable values of the runtime environement as a hashtable (name => value).
	 */
	public Hashtable<String, Object> getVariableValues() {
		return runtimeEnvironment.getVariableValues();
	}
	
	/**
	 * 
	 * Returns all defined variable types in the runtime environment.
	 * 
	 * @return All defined variable tyes of the runtime environement as a hashtable (name => type).
	 */
	public Hashtable<String, Class> getVariableTypes() {
		return runtimeEnvironment.getVariableTypes();
	}

	/**
	 * 
	 * Returns all visited vertices.
	 * 
	 * @return A ArrayList of vertices, which has been entered during the simulation.
	 */
	public ArrayList<Vertex> getVisitedVertices() {
		return visitedVertices;
	}
	
	/**
	 * 
	 * Returns all passes transitions.
	 * 
	 * @return A ArrayList of transitions, which has been passed during the simulation.
	 */
	public ArrayList<Transition> getPassedTransitions() {
		return passedTransitions;
	}
	
	/**
	 * Gets the parameter configuration of a call event.
	 * 
	 * @param callEvent - The MyEvent object
	 * @return All paramteres and values assigned with the MyEvent object.
	 */
	public Hashtable<Parameter, String> getCallEventParameters( MyEvent callEvent ) {
		return parameterConfigurations.get( callEvent );
	}

	/**
	 *
	 * Returns the currently simulated statemachine.
	 * 
	 * @return The statemachine currently simulated.
	 */
	public StateMachine getStateMachine() {
		return sm;
	}
	
	/**
	 * 
	 * Returns the currently simulated simulateable statemachine.
	 * 
	 * @return The simulateable statemachine currently simulated.
	 */
	public AbstractSimulateableStateMachine getSimulateableStateMachine() {
		return assm;
	}
	
	/**
	 * 
	 * Returns the currently active vertices.
	 * 
	 * @return A ArrayList of current active vertices.
	 */	
	public ArrayList<Vertex> getActiveStates() {
		return activeStates;
	}

	/**
	 * 
	 * Returns the next active states.
	 * 
	 * @return A ArrayList of the vertices, which are active in the next round.
	 */
	public ArrayList<Vertex> getNextActiveStates() {
		return nextActiveStates;
	}

	/**
	 * 
	 * Returns the RuntimeEnvironement currently used.
	 * 
	 * @return The currently used RuntimeEnvironement.
	 */
	public MyRuntimeEnvironment getRuntimeEnvironment() {
		return this.runtimeEnvironment;
	}
	
	/**
	 * 
	 * Returns the name of a parameter.
	 * 
	 * @param parameter - The parameter.
	 * @return The name of the parameter as String.
	 */
	public String getParameterName ( Parameter parameter ) {
		return parameter.getName();
	}
	
	/**
	 * 
	 * Returns the type of a parameter. 
	 * 
	 * @param parameter - The parameter
	 * @return The type of the paramter a string.
	 */
	public String getParameterType ( Parameter parameter ) {
		if ( parameter.getType() == null ) return null;
		String parameterType = parameter.getType().getName();
		if (parameterType == null ) return null;

		if (  parameterType.equalsIgnoreCase( "Boolean" ) ) {
			return "Boolean";
		}
		
    	if (  parameterType.equalsIgnoreCase( "Integer" ) ||
    		  parameterType.equalsIgnoreCase( "Int" ) ) {
    		return "Integer";
    	}
    	
    	if (  parameterType.equalsIgnoreCase( "String" ) ) {
    		return "String";
    	}
    	
    	return null;
	}
	
	/**
	 * Returns the value of a parameter. If not set than the default value, 
	 * else a initial value.
	 * 
	 * @param callEvent - The associated event.
	 * @param parameter - The parameter.
	 * @return The value of the parameter or a default value. 
	 */
	public String getParameterValue ( MyEvent callEvent, Parameter parameter ) {
		Hashtable<Parameter, String> parameters = getCallEventParameters( callEvent );
		if ( parameters == null || parameters.get( parameter ) == null ) {

			if ( getParameterType( parameter ) == null ) return "<none>";
			
			if ( parameter.getDefault() != null ) {
				if ( showOutput ) System.out.println( "default");
				return parameter.getDefault();
			} else {
				if ( getParameterType( parameter ).equalsIgnoreCase( "Boolean" ) ) {
					return "false";
				}
				
		    	if ( getParameterType( parameter ).equalsIgnoreCase( "Integer" ) ) {
		    		return "0";
		    	}
		    	
		    	if ( getParameterType( parameter ).equalsIgnoreCase( "String" ) ) {
		    		return "";
		    	}				
			}
		}
		return parameters.get( parameter );
	}
	
	/**
	 * 
	 * Sets the parameter of a call event to value. 
	 * 
	 * @param callEvent - The MyEvent object.
	 * @param parameter - The parameter to be set.
	 * @param value - The value as String.
	 */
	
	public void setParameterValue ( MyEvent callEvent, Parameter parameter, String value ) {
		Hashtable<Parameter, String> parameters = getCallEventParameters( callEvent );
		if ( parameters == null ) {
			parameters = new Hashtable<Parameter, String>();
		}
		parameters.put( parameter, value );
		parameterConfigurations.put( callEvent, parameters );
	}
	
	/**
	 * 
	 * Sets the runtime environement.
	 * 
	 * @param runtimeEnvironement - The new runtime environement to use.
	 */
	public void setRuntimeEnvironement( MyRuntimeEnvironment runtimeEnvironement ) {
		this.runtimeEnvironment = runtimeEnvironement;
	}
	
	/**
	 * 
	 * Sets the parameter configurations. This are used to store the actual
	 * parameter of operations.
	 * 
	 * @param parameterConfigurations - The hashtable to set.
	 */
	public void setParameterConfigurations( Hashtable<MyEvent, Hashtable<Parameter, String>> parameterConfigurations ) {
		this.parameterConfigurations = parameterConfigurations;
	}
	
	/**
	 * 
	 * Returns all parameter configurations. This are used to store the actual
	 * parameter of operations.
	 * 
	 * @return A list whith all MyEvent => Parameter => Value entrys currently set.
	 */
	public Hashtable<MyEvent, Hashtable<Parameter, String>> getParameterConfigurations() {
		return this.parameterConfigurations;
	}
			
	/**
	 * 
	 * Simulates the state machine until one stop condition is true.
	 * 
	 * @return A value of the StopReason enumeration.
	 * @throws NoValidStateMachineException
	 * @throws ExpressionEvaluationException 
	 * @throws ExpressionParseException 
	 */
	public StopReason simulate() throws NoValidStateMachineException, ExpressionParseException, ExpressionEvaluationException {
		stopReason = StopReason.None;
		while ( stopReason == StopReason.None ){
			stopReason = doStep();
		}
		return stopReason;
	}

	/** 
	 *
	 * Prints the elments of the statemachine to stdout.
	 *
	 * @param stateMachine - The statemachine to print.
	 */
	public void PrintDebugOutput( StateMachine stateMachine ) {

		// get regions		
		EList regions = stateMachine.getRegions();
		
		for ( int i = 0; i < regions.size(); i++ ){
			Region region = ( Region ) regions.get( i );
			System.out.println( "Region: " + region.getQualifiedName() );
			// get elements
			EList elements = region.getSubvertices();
			for ( int j = 0; j < elements.size(); j++ ){
				Object element = elements.get( j );			
				if ( element instanceof FinalState ){	
					FinalState state = ( FinalState ) element;
					System.out.println ( "   FinalState: " + state.getQualifiedName() );
				} else if ( element instanceof State ){	
					State state = ( State ) element;
					System.out.println ( "   State: " + state.getQualifiedName() );
				} else if ( element instanceof Pseudostate ){	
					Pseudostate state = ( Pseudostate ) element;
					System.out.println ( "   PseudoState: " + state.getQualifiedName() + " (" + state.getKind() + ")" );
				}			
				
			}
			
			// get transitions
			EList transitions = region.getTransitions();
			for ( int j = 0; j < transitions.size(); j++ ){
				Object element = transitions.get( j );
				if ( element instanceof Transition ){
					Transition trans = ( Transition ) element;
					System.out.println( "      Transtition: " + trans.getQualifiedName() + " : " + trans.getSource().getName() + " --> " + trans.getTarget().getName() );
				}			
			}		
		}	
	}
	
	/**
	 * 
	 * Used to observe the changed variables of the runtime environment. 
	 * Not yet used. 
	 * 
	 */
	public void update( Observable arg0, Object arg1 ) {}
	
	/**
	 * Returns the corresponding string to a StopReason enumeration value.
	 * 
	 * @param stopReason
	 * @return The String describing the StopReason value.
	 */
	public static String StopReasonToString( StopReason stopReason ) {
		switch( stopReason ) {
			case None: return "None";
			case partialDeadlock: return "Partial Deadlock";
			case NoMoreEvents: return "No more events";
			case totalDeadlock: return "Total Deadlock";
		}
		return new String( "Not defined!" );
	}
	
// ======================== PRIVATE PART =======================	
	
	/**
	 * Returns the first element of the list rls.
	 * 
	 * @param rls - The ArrayList of regions where to pop the region.
	 * @return The region on top of the list.
	 */
	private Region popRegionsArrayList( ArrayList<Region> rls ) {
		Region temp;
		try {
			temp = rls.get( 0 );
		} catch ( IndexOutOfBoundsException ex) {
			return null;
		}
		
		if ( temp != null ) {
			rls.remove( temp );
		}
		return temp;
	}
	
	/**
	 * 
	 * Used to set the active vertices of a region and all subregions.
	 * 
	 * @param region - The region, where to start setting the vertices from the historyStates hashtable recursivly.
	 */
	private void setDeepHistoryStatesRec( Region region ) {
		if ( region != null ) {
			EList elements = region.getOwnedElements();
			for ( int i = 0; i < elements.size(); i++ ) {
				Object element = elements.get( i );
				if ( element instanceof Region ) {
					Region temp = ( Region ) element;
					setDeepHistoryStatesRec( temp );
				}
			}
			ArrayList<Vertex> activeElements = historyStates.get( region );
			if ( activeElements != null  ) {
				addNextActiveStatesWithoutBehavior( activeElements );
			}
		}
	}
	
/*
 * ====================== PRIVATE PART ======================
 */	
	
	
	/**
	 * 
	 * Used to store all states in the historyStates hashtable.
	 * 
	 * @param transition - The transition to be added.
	 */
	
	private void transitionPassed( Transition transition ) {

		Vertex source = transition.getSource();
		Vertex target = transition.getTarget();
		
		Region sourceRegion = source.getContainer();
		Region targetRegion = target.getContainer();
		
		if ( showOutput ) System.out.println( "Region1: " + sourceRegion + "\nRegion2: " + targetRegion );

		// Remove the state left from the history states hashtable
		ArrayList<Vertex> temp = historyStates.get( sourceRegion );
		if ( temp != null ) {
			if ( temp.contains( source ) ) {
				temp.remove( source );		
				if ( showOutput ) System.out.println( "HISTORY: STATE REMOVED: " + source);
			}									
		}
		
		// If transition is external, save current active states
		if ( sourceRegion != targetRegion ) {
			if ( showOutput ) System.out.println( "EXTERNAL TRANSITION!");
			// Determine regions path to the root region of both regions
			ArrayList<Region> rl1 = getPathToRoot( sourceRegion );
			ArrayList<Region> rl2 = getPathToRoot( targetRegion );
			// Determine common region in path
			Region common = getCommonRegion( rl1, rl2 );
			// Save states of all regions on the path to the root region until
			// the common region is reached
			if ( showOutput ) System.out.println("--> RL1: " + rl1 );
			if ( showOutput ) System.out.println("--> COMMON: " +  common );
			storeAllActiveStatesInRegionsRec( getAllRegionsUntil( rl1, common ) );
		} else {
			// Transition passed in same region, add the new ones.
			if ( temp == null ) {
				temp = new ArrayList<Vertex>();
				historyStates.put( sourceRegion, temp );
			}
			if ( !temp.contains( target ) ) {
				temp.add( target );		
				if ( showOutput ) System.out.println( "HISTORY: STATE ADDED: " + target );
			}
		}
	}
	
	/**
	 * 
	 * Sets the active states, does the entry/state/exit behavior in the 
	 * correct order, executes the effect of the transition and updates the 
	 * outgoing triggers of a complex state.
	 * 
	 * @param newState - The vertex, which should be active in the next round.
	 * @param oldstate - The vertex, which was active this round.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws NoValidStateMachineException 
	 * @throws NoValidStateMachineException 
	 * @throws NoMoreOutgoingTransitionsException 
	 */
	private void addNextActiveState( Vertex newState, Vertex oldState, Transition transition ) throws ExpressionEvaluationException, ExpressionParseException, NoValidStateMachineException {

		if ( showOutput ) System.out.println( "ADD ACTIVE STATE: " + newState );
		
		// Flag used for the multiple concurrent parallel behavior. Set, if the
		// exit behavior of a state has already been executed.
		boolean exitBehaviorDone = false;
		
		passedTransitions.add( transition );
		// If a complex state is entered, collect the outgoing transitions and
		// if a complex state is left, remove outgoing transitions of the 
		// outgoingsTrantions hashtable
		collectOutgoingTransitionsRec( oldState, newState );
		
		if ( newState instanceof Pseudostate ) {
			Pseudostate ps = ( Pseudostate ) newState;
			if ( showOutput ) System.out.println( "PSEUDOSTATE: " + ps );
		
			// If pseudostate is a join, check, if all incoming states are active  
			if ( ps.getKind() == PseudostateKind.JOIN_LITERAL ) {	
				if ( showOutput ) System.out.println( "JOIN: " + newState );
				ArrayList<Vertex> sources = new ArrayList<Vertex>();
				EList incomings = ps.getIncomings();
				for ( int i = 0; i < incomings.size(); i++ ) {
					Transition incoming = ( Transition ) incomings.get( i );
					Vertex source = incoming.getSource();
					if ( oldState != source ) sources.add( source );
				}
				
				if ( showOutput ) System.out.println( "SOURCES: " + sources );

				if ( !activeStates.containsAll( sources ) ) {
					// Not all incoming states are active ==> add old state
					nextActiveStates.add( oldState );
					return;
				} else {
					// All incoming states are active ==> remove states and add
					// outgoing state of join.
					activeStates.removeAll( sources );

					if ( multipleConcurrentBehavior == MultipleConcurrentBehavior.sequential ) {
						// Do behavior on path sequential
						for ( int i = 0; i < sources.size(); i++ ) {
							Vertex source = sources.get( i );
							if ( source instanceof State ) {
								doExit( ( State ) source );
							}	
							Transition _transition = getTransitionBetween( source, newState );
							doBehaviorSequential( source, newState, _transition  );
							transitionPassed( _transition );
						}
					} else {
						// Do behavior pseudo parallel						

						// 1st, do the exit behavior of the source states
						// and add them to the visitedVertices list.
						for ( int i = 0; i < incomings.size(); i++ ) {
							transition = ( Transition ) incomings.get( i );
							Vertex source = transition.getSource();
							if ( source instanceof State ) {
								doExit( ( State ) source );
							}
							// visitedVertices.add( source );
						}
						
						exitBehaviorDone = true;
						
						ArrayList<ArrayList<Region>> rls = new ArrayList<ArrayList<Region>>();
						ArrayList<Region> rl1 = getPathToRoot( newState.getContainer() );
						for ( int i = 0; i < incomings.size(); i++ ) {
							transition = ( Transition ) incomings.get( i );
							Vertex source = transition.getSource();							
							ArrayList<Region> rl2 = getPathToRoot( source.getContainer() );
							Region common = getCommonRegion( rl1 , rl2 );
							rls.add( getAllRegionsUntil( rl2, common ) );
							//addNextActiveState( target, currentVertex, transition );
							passedTransitions.add( transition );
							transitionPassed( transition );
							// If a complex state is entered, collect the outgoing transitions and
							// if a complex state is left, remove outgoing transitions of the 
							// outgoingsTrantions hashtable
							collectOutgoingTransitionsRec( source, newState );
						}
						
						if ( showOutput ) System.out.println( "COMMON: " + rls );
						
						Region sourceRegion = newState.getContainer();

						int max = getMaxRegionsInList( rls );
						int count = rls.size() * max;
						// Go throug all lists an pick the region on top of
						// each list.
						for ( int i = 0; i < count; i++ ) {
							ArrayList<Region> temp = rls.get( i % rls.size() );
							Region region = popRegionsArrayList( temp );
							if ( region != null ) {
								if ( sourceRegion != region ) {
									// Store active region (not used yet).
									activeRegions.remove( sourceRegion );
									activeRegions.add( region );
								}
								State state = region.getState();
								if ( state != null ) {
									doExit( state );
									visitedVertices.add( state );
								}
							}
						}
					}
				}
			}
		}
		
		if ( oldState instanceof State && !exitBehaviorDone ) {
			doExit( ( State ) oldState );
		}		
			
		// Test, if transition points to a state within a complex state and do
		// behavior sequential.
		Region sourceRegion = oldState.getContainer();
		Region targetRegion = newState.getContainer();
		
		if ( multipleConcurrentBehavior == MultipleConcurrentBehavior.sequential )
			doBehaviorSequential( oldState, newState, transition );
		
		// Store active region (not used yet).
		activeRegions.remove( sourceRegion );
		activeRegions.add( targetRegion );
		
		// Update history states
		transitionPassed( transition );

		visitedVertices.add( newState );

		if ( newState instanceof State ) {
			State state = ( State ) newState;
			doEntry( state );
			doDo( state );
			if ( state.isComposite() || state.isOrthogonal() ) {
				// Needed to fix problem with normal states marked as complex state
				boolean complexDefaultEntry = false;
				EList regions = state.getRegions();
				// Collect the initial states in all regions and set them as active
				// next round.
				for ( int i = 0; i < regions.size(); i++ ) {
					Region region = ( Region ) regions.get( i );
					EList elements = region.getOwnedElements();
					for ( int j = 0; j < elements.size(); j++ ) {
						Object element = elements.get( j );
						if ( element instanceof Pseudostate ) {
							Pseudostate ps = ( Pseudostate ) element;
							if ( ps.getKind() == PseudostateKind.INITIAL_LITERAL ) {
								complexDefaultEntry = true;
								visitedVertices.add( ps );
								nextActiveStates.add( ps );							
								if ( showOutput ) System.out.println( "Initial: " + ps );
							}
						}
					}
				}
				if ( complexDefaultEntry ) {
					if ( showOutput ) System.out.println( "--> DEFAULT ENTRY");
					// Stores all outgoing transitions in the hashtable
					EList outgoings = state.getOutgoings();
					if (  outgoings != null && outgoings.size() > 0 ) {
						externalTransitions.put( state, outgoings );
					}		
					return;
				}
			}			
		}
		
		nextActiveStates.add( newState );
	}
	
	/**
	 * 
	 * Returns the transition between the source vertex and the target vertex, 
	 * or null if no such transition exists. 
	 * 
	 * @param source - The source vertex.
	 * @param target - The target vertex.
	 * @return The transition between the two vertices.
	 */
	private Transition getTransitionBetween( Vertex source, Vertex target ) {
		EList outgoings = source.getOutgoings();
		for ( int i = 0; i < outgoings.size(); i++ ) {
			Transition transition = ( Transition ) outgoings.get( i );
			if ( transition.getTarget() == target ) return transition;
		}		
		return null;
	}
	
	
	/**
	 * 
	 * Adds a state to the activeStates list, not the nextActiveStates list
	 * 
	 * @param newState - The state to be added.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws NoValidStateMachineException 
	 */
	
	private void addActiveState( Vertex newState ) throws ExpressionEvaluationException, ExpressionParseException, NoValidStateMachineException {

		// activeStates.remove( oldState );
		activeStates.add( newState );
		visitedVertices.add( newState );
				
		if ( !( newState instanceof Pseudostate ) && newState instanceof State ) {
			doEntry( ( State ) newState );
		}
	}	
	
	/**
	 * 
	 * Adds a vertice without performing entry and exit behavior.
	 * 
	 * @param state - The state to be active in the next round.
	 */
	private void addNextActiveStateWithoutBehavior ( Vertex newState ) {		
		if ( !nextActiveStates.contains( newState ) ) {
			nextActiveStates.add( newState );
		}
	}
	
	/**
	 * 
	 * Adds a couple of vertices without performing entry and exit beahavior.
	 * 
	 * @param states - A ArrayList of vertices.
	 */
	private void addNextActiveStatesWithoutBehavior ( ArrayList<Vertex> states ) {
		if ( !nextActiveStates.containsAll( states ) ) {
			nextActiveStates.addAll( states );
		}
		visitedVertices.addAll( states );
	}
	
	/**
	 * 
	 * Does the entry behavior of the state.
	 * 
	 * @param state - The state, where to do the behavior.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws NoValidStateMachineException 
	 */
	private void doEntry( State state ) throws ExpressionEvaluationException, ExpressionParseException, NoValidStateMachineException {
		if ( state == null ) return;
		Behavior behavior = state.getEntry();
		if ( behavior == null ) return;
		if ( showOutput ) System.out.print( "Entry behavior: " );
		doBehavior( behavior );
	}
	
	/**
	 * 
	 * Does the exit behavior of state.
	 * 
	 * @param state - The state, where to do the behavior.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws NoValidStateMachineException 
	 */
	private void doExit( State state ) throws ExpressionEvaluationException, ExpressionParseException, NoValidStateMachineException {
		Behavior behavior = state.getExit();
		if ( behavior == null ) return;
		if ( showOutput ) System.out.print( "Exit behavior: " );
		doBehavior( behavior );
	}

	/**
	 * 
	 * Does the do activity of state.
	 * 
	 * @param state - The state, where to do the behavior.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws NoValidStateMachineException 
	 */
	private void doDo( State state ) throws ExpressionEvaluationException, ExpressionParseException, NoValidStateMachineException {
		Behavior behavior = state.getDoActivity();
		if ( behavior == null ) return;
		System.out.print( "Do behavior: " );
		doBehavior( behavior );
	}
	
	/**
	 * 
	 * Does the behavior of the operation.
	 * 
	 * @param callEvent - The event used to map parameters to values or null, if not parameters available.
	 * @param operation - The operation, where to do the behavior.
	 * @throws ExpressionParseException 
	 */	
	private void doOperation( MyEvent callEvent, Operation operation ) throws ExpressionEvaluationException, ExpressionParseException {
		
		ArrayList<String> temp_parameters = new ArrayList<String>();
		// Set local paramters according to paramters map
		EList parameters = operation.getOwnedParameters();
		if ( callEvent != null ) {
			for ( int i = 0; i < parameters.size(); i++ ) {
				Parameter parameter = ( Parameter) parameters.get( i );
				if ( !(parameter.getDirection() == ParameterDirectionKind.IN_LITERAL ||
					 parameter.getDirection() == ParameterDirectionKind.INOUT_LITERAL) ) continue;
	
				String parameterValue = getParameterValue( callEvent, parameter );  	
				String parameterType = getParameterType( parameter );
				// If name of parameter no set, use type and position of the 
				// parameter as name, e.g. Integer:0 
				String parameterName = ( getParameterName( parameter ) != null ) ? getParameterName( parameter ) : parameterType + ":" + i ;
				
				if ( showOutput ) System.out.println( "PARAMETER CONFIGURATION: " + parameterName + " : " + parameterValue + " : " + parameterType );
				
				runtimeEnvironment.setVariableType( parameterName, MyRuntimeEnvironment.getClassFromType( parameterType ) );
				runtimeEnvironment.setVariable( parameterName, MyRuntimeEnvironment.getObjectFromType( runtimeEnvironment.getVariableType( parameterName ), parameterValue  )  );
				temp_parameters.add ( parameterName );
			}
		}
		
		// Evaluate all expressions in the contraint body
		EList rules = operation.getOwnedRules();
		for ( int j = 0; j < rules.size(); j++ ) {
			Constraint constraint = ( Constraint ) rules.get( j );
			ValueSpecification specification = constraint.getSpecification();
			if ( specification instanceof OpaqueExpression ) {
				OpaqueExpression expression = ( OpaqueExpression ) specification;
				EList bodies = expression.getBodies();
				for ( int k = 0; k < bodies.size(); k++ ) {
					String exp = ( String ) expression.getBodies().get( k );
					if ( k > expression.getLanguages().size() - 1 ) continue;
					String lang = ( String ) expression.getLanguages().get( k );
					// Ignore not suppoerted languages
					if ( lang.equalsIgnoreCase( "Expression" ) || lang.equalsIgnoreCase( "OCL" ) 
							|| lang.equalsIgnoreCase( "OCL1.5" ) || lang.equalsIgnoreCase( "OCL2.0" ) )  {
						// Parse the expresions ...
						de.cnc.expression.Expression expparser = de.cnc.expression.Expression.parse( exp , runtimeEnvironment );
						// ... and evaluate it with the variables set in the runtime environement.
						expparser.eval( runtimeEnvironment );	
						if ( showOutput ) System.out.println( "Operation [" + operation.getName() + "]: " + exp + " evaluated." );
					} else {
						if ( showOutput ) System.out.println( "Expresion of type \"" + lang + "\" ignored.");
					}
				}
			}
		}
		
		// Remove variables temporaily added to the runtime environement. 
		for ( int i = 0; i < temp_parameters.size(); i++ ) {
			runtimeEnvironment.removeVariable( temp_parameters.get( i ) );
		}	
	}
	
	/**
	 * 
	 * Simulates another statemachine as behavior. The options and active events 
	 * are set according to the "calling" statemachine.  
	 * 
	 * <b>Note:</b> Not yet carefully tested!
	 * 
	 * @param stateMachine - The stahemachine to subsimulate.
	 * @throws NoValidStateMachineException 
	 * @throws ExpressionEvaluationException 
	 * @throws ExpressionParseException 
	 */
	private void doStateMachine( StateMachine stateMachine ) throws NoValidStateMachineException, ExpressionParseException, ExpressionEvaluationException {
		SimulateableStateMachine ssm = ( SimulateableStateMachine ) assm;
		Model model = ssm.getModel();
		SimulateableStateMachine sssm = SimulateableStateMachine.getSimulateableStateMachine( model, stateMachine );
		Simulator subsim = new Simulator( sssm );
		subsim.setupSimulation();
		
		// Set parameters according to calling simulator instance.
		subsim.setStopIfNoMoreEvents( this.getStopIfNoMoreEvents() );
		subsim.setStopIfPartialDeadlock( this.getStopIfPartialDeadlock() );
		subsim.setStopIfTotalDeadlock( this.getStopIfTotalDeadlock() );
		subsim.setActiveEvents( this.getActiveEvents() );
		subsim.setEventQueueBehavior( this.getEventQueueBehavior() );
		subsim.setMultipleConcurrentBehavior( this.getMultipleConcurrentBehavior() );
		subsim.setParameterConfigurations( this.getParameterConfigurations() );
		subsim.setRuntimeEnvironement( this.getRuntimeEnvironment() );		
		
		this.stopReason = subsim.simulate();
	}
	
	/**
	 *
	 * Does the effect associated with the transition
	 * 
	 * @param transition - The transition.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws NoValidStateMachineException 
	 */
	private void doEffect( Transition effect ) throws ExpressionEvaluationException, ExpressionParseException, NoValidStateMachineException {
		if ( effect == null ) return;
		doBehavior( effect.getEffect() );
	}

	/**
	 * 
	 * Does the behavior. Currently supported is behavior specified as operation
	 * or statemachine (not yet carefully tested). 
	 * 
	 * @param behavior - The behavior to be executed.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws NoValidStateMachineException 
	 */
	private void doBehavior( Behavior behavior ) throws ExpressionEvaluationException, ExpressionParseException, NoValidStateMachineException {
		
		if ( behavior == null ) return;
		
		String behaviorName = behavior.getName();
		if ( showOutput ) System.out.println( "Behavior: " + behaviorName );
		
		if ( behavior instanceof Interaction ) {}
		if ( behavior instanceof StateMachine ) {
			StateMachine stateMachine = ( StateMachine ) behavior;
			doStateMachine( stateMachine );
		}
		// Used by Magic Draw for interaction instead of a interaction object
		if ( behavior instanceof Activity ) {
			Activity activity = ( Activity ) behavior;
			EList operations = activity.getAllOperations();
			for ( int i = 0; i < operations.size(); i++ ) {
				Operation operation = ( Operation ) operations.get( i );
				doOperation( null, operation );
			}
		}
	}
		
	/**
	 * 
	 * Checks the guard for the transition transition. If a MyEvent object
	 * is passed, the actual parameters assiciated with the object are also
	 * added to the runtime environement temporarily.
	 * 
	 * @param transition - The transition, where to check the guard expression.
	 * @return boolean - True, if the guard expression is true or false.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws GuardNotOfTypeExpression 
	 */
	private boolean checkGuard( Transition transition, MyEvent event ) throws ExpressionParseException, ExpressionEvaluationException {
		
		Constraint guard = transition.getGuard();
		// If no guard exists, transition is allowed to be traversed
		if ( guard == null ) return true;
		
		ArrayList<String> temp_parameters = new ArrayList<String>();
		// Add parameters of event to runtime environement
		if ( event != null && event.getEventType() == EventType.callOperation ) {
			Operation operation = event.getOperation();
			EList parameters = operation.getOwnedParameters();
			for ( int i = 0; i < parameters.size(); i++ ) {
				Parameter parameter = ( Parameter) parameters.get( i );
				if ( !(parameter.getDirection() == ParameterDirectionKind.IN_LITERAL ||
					 parameter.getDirection() == ParameterDirectionKind.INOUT_LITERAL) ) continue;

				String parameterValue = getParameterValue( event, parameter );  	
				String parameterType = getParameterType( parameter );
				
				// If name of parameter no set, use type and position of the 
				// parameter as name, e.g. Integer:0 
				String parameterName = ( getParameterName( parameter ) != null ) ? getParameterName( parameter ) : parameterType + ":" + i ;				
				runtimeEnvironment.setVariableType( parameterName, MyRuntimeEnvironment.getClassFromType( parameterType ) );
				runtimeEnvironment.setVariable( parameterName, MyRuntimeEnvironment.getObjectFromType( runtimeEnvironment.getVariableType( parameterName ), parameterValue  )  );
				temp_parameters.add ( parameterName );			}
		}
		
		ValueSpecification specification = guard.getSpecification();
		boolean b = false;
		// Check the expressions of the guard body. 
		if ( specification instanceof OpaqueExpression ) {
			OpaqueExpression expression = ( OpaqueExpression ) specification;
			for ( int i = 0; i < expression.getBodies().size(); i++ ) {
				String exp = ( String ) expression.getBodies().get( i );
				String lang = ( String ) expression.getLanguages().get( i );
				System.out.print( "checkGuard[" + lang + "]: " + exp + " => " );
				if ( lang.equalsIgnoreCase( "Expression" ) || lang.equalsIgnoreCase( "OCL" ) 
						|| lang.equalsIgnoreCase( "OCL1.5" ) || lang.equalsIgnoreCase( "OCL2.0" ) )  {
					if ( exp.equalsIgnoreCase( "else" ) ) return false;
					de.cnc.expression.Expression expParser = de.cnc.expression.Expression.parse( exp , runtimeEnvironment );
					Object result = expParser.eval( runtimeEnvironment );
					if ( !(result instanceof Boolean) ) {
						if ( showOutput ) System.out.println( "Expression exception in guard [" + exp + "]: Guard does not return a boolean value" );
					} else {
						// If one of the expresions is true, the guard is true. 
						b = b || ( ( Boolean ) result ).booleanValue();
						if ( showOutput ) System.out.println( b );
					}	
					
				} else {
					if ( showOutput ) System.out.println( "Guard-Expresion not of type \"Expression\" or \"OCL\"! Ignored.");
				}
			}			
		}

		
		// Remove variables added temporarily. 
		for ( int i = 0; i < temp_parameters.size(); i++ ) {
			runtimeEnvironment.removeVariable( temp_parameters.get( i ) );
		}
		
		return b;
	}
	
	/**
	 * 
	 * Checks, if a event for a trigger is available according to the event list
	 * behavior and if the event should be removed.
	 * 
	 * @param event - The EMF UML2 event to be checked against the events in the event queue.
	 * @return MyEvent - The MyEvent object of the events queue assiciated with the call or the signal event. Null, if event not available.
	 * @throws ExpressionEvaluationException 
	 * @throws ExpressionParseException 
	 */
	private MyEvent checkTrigger( Event event ) throws ExpressionEvaluationException, ExpressionParseException {	

		if ( thisRoundActiveEvent != null ) {	

			if ( event instanceof CallEvent ) {
				// If event is a callEvent, check the queue, if a MyEvent object
				// is on top of the queue with the same operation.
				CallEvent callEvent = ( CallEvent ) event;
				Operation operation = callEvent.getOperation();
				if ( thisRoundActiveEvent.getEventType() == EventType.callOperation &&
						thisRoundActiveEvent.getOperation() == operation ) {
					// Do associated operation 
					doOperation( thisRoundActiveEvent, operation );
					//thisRoundActiveEvent = null
					//nextUnActiveEvents.add ( myEvent );
					return thisRoundActiveEvent;
				}
			}
		
			if ( event instanceof SignalEvent ) {
				// If event is a signalEvent, check the queue, if a MyEvent object
				// is on top of the queue with the same signal.				
				SignalEvent signalEvent = ( SignalEvent ) event;
				Signal signal = signalEvent.getSignal();
				if ( thisRoundActiveEvent.getEventType() == EventType.sendSignal &&
						thisRoundActiveEvent.getSignal() == signal ) {
					// activeEvents.remove( myEvent );
					//nextUnActiveEvents.add ( myEvent );
					return thisRoundActiveEvent;
				}
			}
		}

		if ( eventsQueueBehavior == EventsQueueBehavior.skip ) {
			// Skip all events until event is found or end of queue.
			for ( int i = 0; i < activeEvents.size(); i++ ) {
				MyEvent _myEvent = activeEvents.get( i );
				if ( event instanceof CallEvent ) {
					CallEvent callEvent = ( CallEvent ) event;
					Operation operation = callEvent.getOperation();
					if ( _myEvent.getEventType() == EventType.callOperation && _myEvent.getOperation() == operation ) {
						// Do associated operation 
						doOperation( _myEvent, operation );
						activeEvents.remove( _myEvent );
						return _myEvent;
					}
				}
				
				if ( event instanceof SignalEvent ) {
					SignalEvent signalEvent = ( SignalEvent ) event;
					Signal signal = signalEvent.getSignal();
					if ( _myEvent.getEventType() == EventType.sendSignal && _myEvent.getSignal() == signal ) {
						activeEvents.remove( _myEvent );
						return _myEvent;
					}
				}						
			}
		}			
		
		// If event not on top of queue or defered, return null.
		return null;
	}
	
	/**
	 * 
	 * Does the entry & exit behavior sequential from source to target.
	 * 
	 * @param source - The source region.
	 * @param target - The target region.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws NoValidStateMachineException 
	 */
	private void doBehaviorSequential ( Vertex source, Vertex target, Transition transition ) throws ExpressionEvaluationException, ExpressionParseException, NoValidStateMachineException {
		
		Region r1 = source.getContainer();
		Region r2 = target.getContainer();
		
		if ( r1 == r2 ) return;
		
		if ( showOutput ) System.out.println( "EXPLICIT ENTRY/EXIT");
		ArrayList<Region> rl1 = getPathToRoot( r1 );
		ArrayList<Region> rl2 = getPathToRoot( r2 );
		Region common = getCommonRegion( rl1, rl2 );		
		
		doExitBehaviorRec( rl1, common );
		doEffect ( transition );
		doEntryBehaviorRec( rl2, common );			
	}
	
	/**
	 * 
	 * Does the exit behavior of the complete states path.
	 * 
	 * @param rl - The list of regions, where to do the exit behavior.
	 * @param common - The region, until the behavior should be executed.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws NoValidStateMachineException 
	 */
	private void doExitBehaviorRec( ArrayList<Region> rl, Region common ) throws ExpressionEvaluationException, ExpressionParseException, NoValidStateMachineException {
		for ( int i = 0; i < rl.size(); i++ ) {
			Region region = rl.get( i );
			if ( region == common ) return;
			activeRegions.remove( region );
			State state = region.getState();
			if ( state != null ) {
				doExit( state );
			}
		}
	}
	
	/**
	 * 
	 * Does the entry beahvior of the complete states path.
	 * 
	 * @param rl - The list of regions, where to do the entry behavior.
	 * @param common - The region, until the behavior should be executed.
	 * @throws ExpressionParseException 
	 * @throws ExpressionEvaluationException 
	 * @throws NoValidStateMachineException 
	 */
	private void doEntryBehaviorRec( ArrayList<Region> rl, Region common ) throws ExpressionEvaluationException, ExpressionParseException, NoValidStateMachineException {
		
		int index = rl.indexOf( common );
		for ( int i = index - 1; i > -1; i-- ) {
			Region region = rl.get( i );
			if ( region == common ) return;
			activeRegions.remove( region );
			State state = region.getState();
			if ( state != null ) {
				doEntry( state );
				visitedVertices.add( state );
			}
		}
	}
	
	/**
	 * 
	 * Collects/removes recursivly all outgoings transitions.
	 * 
	 * @param source - The source state from where to collect the transitions.
	 * @param target - The target state to where to collect the transitions.
	 */
	private void collectOutgoingTransitionsRec( Vertex source, Vertex target ) {
		if ( source == null ) return;
		if ( target == null ) return;
				
		if ( target instanceof State ) {
			State state = ( State ) target;
			if ( state.isOrthogonal() || state.isComposite() ) {
				externalTransitions.put( state, state.getOutgoings() );
			}
		}	
		
		// Collect all regions from state to root
		ArrayList<Region> sourceRegionsLeft = getPathToRoot( source.getContainer() );
		ArrayList<Region> targetRegionsEntered = getPathToRoot( target.getContainer() );
		// get common region to be excluded
		Region common = getCommonRegion( sourceRegionsLeft, targetRegionsEntered );
		
		// Remove complex states left from the external transitons list
		for ( int i = 0; i < sourceRegionsLeft.size(); i++ ) {
			Region region = sourceRegionsLeft.get( i );
			if ( region == common ) break;
			State state = region.getState();
			if ( state != null ) {
				externalTransitions.remove( state );
				historyStates.remove( state );
			}	
		}
		
		// And complex states entered
		for ( int i = 0; i < targetRegionsEntered.size(); i++ ) {
			Region region = targetRegionsEntered.get( i );
			if ( region == common ) break;
			State state = region.getState();
			if ( state != null ) {
				externalTransitions.put( state, state.getOutgoings() );
			}
		}

	}
	
	/**
	 * Returns the regions between region "region" and the root region.
	 * 
	 * @param region - The source region.
	 * @return The ArrayList of regions between region and the root region.
	 */
	private ArrayList<Region> getPathToRoot( Region region ) {
		ArrayList<Region> path = new ArrayList<Region>();
		getPathToRootRec( region, path );
		return path;
	}
	
	/**
	 * Used by getPathToRoot() to collect the all regions to the root region. 
	 * 
	 * @param region - The source region.
	 * @param path - The path from region to the root region.
	 */
	private void getPathToRootRec( Region region, ArrayList<Region> path ) {
		path.add( region );
		State state = region.getState();
		// If region is associated to a state and not to the statemachine 
		// (= root region), go into recusion.
		if ( state != null ) {
			getPathToRootRec( state.getContainer(), path );
		}
	}
	
	/**
	 * 
	 * Returns the first common region of regions list of l1 and rl2.
	 * 
	 * @param rl1 - A first list if regions.
	 * @param rl2 - A second list of regions.
	 * @return The first common region of both region lists.
	 */
	private Region getCommonRegion( ArrayList<Region> rl1, ArrayList<Region> rl2 ) {
		for ( int i = 0; i < rl1.size(); i++ ) {
			Region region = rl1.get( i );
			if ( rl2.contains( region ) ) {
				return region;
			}
		}
		return null;
	}

	/**
	 * 
	 * Return all regions in the list until "until" region. 
	 * 
	 * @param rl - A list of regions.
	 * @param until - The region where to stop.
	 * @return A list of regions from rl until the region is reached.
	 */
	private ArrayList<Region> getAllRegionsUntil( ArrayList<Region> rl, Region until ) {
		ArrayList<Region> temp = new ArrayList<Region>();
		for ( int i = 0; i < rl.size(); i++ ) {
			Region region = rl.get( i );
			if ( region == until ) {
				return temp;
			}
			temp.add( region );
		}
		return temp;
	}
	
	/**
	 * Returns the number of regions of the list with the maximum regions.
	 * 
	 * @param rls - A list if region lists.
	 * @return The maximum number of regions in all lists.
	 */
	private int getMaxRegionsInList( ArrayList<ArrayList<Region>> rls ) {
		int max = -1;
		for ( int i = 0; i < rls.size(); i++ ) {
			ArrayList<Region> regions = rls.get( i );
			max = ( regions.size() > max ) ? regions.size() : max;
		}
		return max;
	}
	
	/**
	 * Saves all active states of all regions of the rl regions to the
	 * historyStates hashtable.  
	 * 
	 * TODO: to be tested
	 * 
	 * @param rl - The list of regions where to save all states to the historyStates hashtable.
	 */
	private void storeAllActiveStatesInRegionsRec( ArrayList<Region> rl) {
		// Go trough all regions in the list
		for ( int i = 0; i < rl.size(); i++ ) {
			if ( showOutput ) System.out.println( "--> " + i );
			Region region = rl.get( i );
			EList elements = region.allOwnedElements();
			for ( int j = 0; j < elements.size(); j++ ) {
				Object object = elements.get( j );
				if ( object instanceof Vertex ) {
					Vertex vertex2 = ( Vertex ) object;
					// If state was active or would be active next round
					// add it to the history states hashtable
					if ( activeStates.contains( vertex2 ) || nextActiveStates.contains( vertex2 ) ) {
						ArrayList<Vertex> temp = historyStates.get( region );
						if ( temp == null ) {
							temp = new ArrayList<Vertex>();
							historyStates.put( region, temp );
						}
						if ( !temp.contains( vertex2 ) ) {
							temp.add( vertex2 );						
						}
					}
				}
			}
		}
	}
	
	/**
	 * 
	 * Adds a region to the the active regions list (not yet used).
	 * 
	 * @param region - The region to be added.
	 */
	private void addActiveRegion( Region region ) {
		activeRegions.add( region );
	}	
	
	
	/**
	 * 
	 * Inverst a ArrayList of regions.
	 * 
	 * @param rls - The ArrayList of regions to be inverted.
	 * @return The inverted ArrayList of regions.
	 */
	private ArrayList<Region> invertRegionsList( ArrayList<Region> rls ) {
		ArrayList<Region> temp = new ArrayList<Region>();
		for ( int i = rls.size() - 1; i > -1; i-- ) {
			Region region = rls.get( i );
			temp.add( region );
		}
		return temp;
	}

}
