package de.fau.cs.swe.da.modelsimulator;

import org.eclipse.uml2.uml.*;
import org.eclipse.emf.common.util.*;

import de.cnc.expression.exceptions.ExpressionEvaluationException;

import java.math.BigDecimal;
import java.util.*;
import java.util.Enumeration;

/**
 * Class, used to simulate state machine, which will be built from a UML2 model.
 * 
 * @author Dominik Schindler
 */

public class SimulateableStateMachine extends AbstractSimulateableStateMachine {

	private static Hashtable<String, StateMachine> s_stateMachines = null;
	private static Hashtable<String, Property> s_ownedVariables = null;
	private static ArrayList<Event> s_availableEvents = null;
	private static MyRuntimeEnvironment s_runtimeEnvironment = null;

	private Model model = null;
	private StateMachine stateMachine = null;
	private ArrayList<Event> availableEvents = null;
	private MyRuntimeEnvironment runtimeEnvironment = null;
	
	@Override
	public StateMachine getStateMachine() {
		return stateMachine;
	}

	@Override
	public MyRuntimeEnvironment initalRuntimeEnvironement() {
		return runtimeEnvironment;
	}
	
	@Override
	public ArrayList<MyEvent> getAvailableEvents() {
		ArrayList<Operation> operations = new ArrayList<Operation>();
		ArrayList<Signal> signals = new ArrayList<Signal>();
		ArrayList<MyEvent> events = new ArrayList<MyEvent>();
		
		for ( int i = 0; i < availableEvents.size(); i++ ) {
			Event event = availableEvents.get( i );
			if ( event instanceof CallEvent ) {
				CallEvent callEvent = ( CallEvent ) event;
				Operation operation = callEvent.getOperation();
				if ( operation != null && !operations.contains( operation ) ) {
					operations.add( operation );
				}
			}
			if ( event instanceof SignalEvent) {
				SignalEvent signalEvent = ( SignalEvent ) event;
				Signal signal = signalEvent.getSignal();
				if ( signal != null && !signals.contains( signal ) ) {
					signals.add( signal );
				}
			}
		}
		
		for ( int i = 0; i < operations.size(); i++ ) {
			Operation operation = operations.get( i );
			MyEvent myEvent = new MyEvent( operation );
			events.add( myEvent );
		}
		
		for ( int i = 0; i < signals.size(); i++ ) {
			Signal signal = signals.get( i );
			MyEvent myEvent = new MyEvent( signal );
			events.add( myEvent );
		}
		
		return events;
	}
	
	/**
	 * Returns the model associated with the simulateable state machine.
	 * 
	 * @return The UML2 model.
	 */
	public Model getModel() {
		return model;
	}
	
	/**
	 * Constructor, which buils a simualteable state machine object.
	 * 
	 * @param model - The UML2 model.
	 * @param stateMachine - The state machine, where to build a simualteable statemachine of.
	 * @param availableEvents - The list of all available events in the model.
	 * @param runtimeEnvironment - The initial runtime environment used for the expression parser.
	 */
	public SimulateableStateMachine( Model model, StateMachine stateMachine, 
			ArrayList<Event> availableEvents, MyRuntimeEnvironment runtimeEnvironment ) {
		this.model = model;
		this.stateMachine = stateMachine;
		this.availableEvents = availableEvents;
		this.runtimeEnvironment = runtimeEnvironment;
	}
	
	public String toString() {
		return stateMachine + " : " + availableEvents;
	}
	
	/**
	 * 
	 * Build the runtime environement according to the defined variables 
	 * in the class.
	 * 
	 * @throws ExpressionEvaluationException 
	 *
	 */
	
	public static void buildRuntimeEnvironement() throws ExpressionEvaluationException {
	   	// build runtime enviroment from defined variables
		Enumeration enumeration = s_ownedVariables.keys();
        while ( enumeration.hasMoreElements() ) {
        	Object key = enumeration.nextElement();
        	Property property = s_ownedVariables.get( key );
        	String type = property.getType().getName();
        	
        	if ( MyRuntimeEnvironment.getClassFromType( type ) == Boolean.class ) {
				s_runtimeEnvironment.setVariableType( property.getName(), MyRuntimeEnvironment.getClassFromType( type ) );
				if ( property.isSetDefault() )  {
					s_runtimeEnvironment.setVariable( property.getName(), new Boolean( Boolean.parseBoolean( property.getDefault() ) ) );
				} else {
					s_runtimeEnvironment.setVariable( property.getName(), new Boolean( false ) );
				}
        	}
        	if ( MyRuntimeEnvironment.getClassFromType( type ) == BigDecimal.class ) {
           		s_runtimeEnvironment.setVariableType( property.getName(), MyRuntimeEnvironment.getClassFromType( type ) );
				if ( property.isSetDefault() )  {
					s_runtimeEnvironment.setVariable( property.getName(), new BigDecimal( Integer.parseInt( property.getDefault() ) ) );
				} else {
					s_runtimeEnvironment.setVariable( property.getName(), new BigDecimal( 0 ) );
				}
        	}
        	if ( MyRuntimeEnvironment.getClassFromType( type ) == String.class ) {
        		s_runtimeEnvironment.setVariableType( property.getName(), MyRuntimeEnvironment.getClassFromType( type ) );
				if ( property.isSetDefault() )  {
					s_runtimeEnvironment.setVariable( property.getName(), property.getDefault() );
				} else {
					s_runtimeEnvironment.setVariable( property.getName(), new String ("") );
				}
        	}
        }
		
	}
	
	/**
	 * 
	 * Goes through the whole model, collects the state machines and puts
	 * them into the hash table.
	 * 
	 * @param packageableElement - The overall packagable element to analyze.
	 */

	private static void collectStateMachinesRec ( PackageableElement packageableElement ) {
		EList packageableElements = null;	
		
		if ( packageableElement instanceof Model ) {
			Model  model = ( Model ) packageableElement;
			packageableElements = model.getPackagedElements();			
		}
		
		if ( packageableElement instanceof org.eclipse.uml2.uml.Package ) {
			org.eclipse.uml2.uml.Package pack = ( org.eclipse.uml2.uml.Package ) packageableElement;
			packageableElements = pack.getPackagedElements();			
		}
		
		if ( packageableElement instanceof org.eclipse.uml2.uml.Class ) {
			org.eclipse.uml2.uml.Class c = ( org.eclipse.uml2.uml.Class ) packageableElement;
			EList behaviors = c.getOwnedBehaviors();
			for ( int i = 0; i < behaviors.size(); i++ ) {
				Behavior behavior = ( Behavior ) behaviors.get( i );
				if ( behavior != null && behavior instanceof StateMachine ) {
					s_stateMachines.put( behavior.getQualifiedName(), ( StateMachine ) behavior );
				}
			}
		}		

		if ( packageableElements == null) return;
		
		for ( int i = 0; i < packageableElements.size(); i++ ) {
			PackageableElement packageableElement2 = ( PackageableElement ) packageableElements.get( i );
			collectStateMachinesRec( packageableElement2 );
		}			
		
	}
	
	/**
	 * 
	 * Goes through the whole model, collects the defined variables and puts
	 * them into the hash table.
	 * 
	 * @param packageableElement - The overall packagable element to analyze.
	 */

	private static void collectOwnedVariablesRec( PackageableElement packageableElement ) {
		EList packageableElements = null;	
		
		if ( packageableElement instanceof Model ) {
			Model  model = ( Model ) packageableElement;
			packageableElements = model.getPackagedElements();			
		}
		
		if ( packageableElement instanceof org.eclipse.uml2.uml.Package ) {
			org.eclipse.uml2.uml.Package pack = ( org.eclipse.uml2.uml.Package ) packageableElement;
			packageableElements = pack.getPackagedElements();			
		}
		
		if ( packageableElement instanceof org.eclipse.uml2.uml.Class ) {
			org.eclipse.uml2.uml.Class c = ( org.eclipse.uml2.uml.Class ) packageableElement;
			EList attributes = c.getOwnedAttributes();
			for ( int i = 0; i < attributes.size(); i++ ) {
				Property property = ( Property ) attributes.get( i );
				if ( property != null && property.getName() != null &&
						property.getType() != null ) {
					s_ownedVariables.put ( property.getName(), property );
				}
			}
		}		

		if ( packageableElements == null) return;
		
		for ( int i = 0; i < packageableElements.size(); i++ ) {
			PackageableElement packageableElement2 = ( PackageableElement ) packageableElements.get( i );
			collectOwnedVariablesRec( packageableElement2 );
		}			
	}
	
	/**
	 * 
	 * Goes through the whole model, collects the available events and puts
	 * them into the list.
	 * 
	 * @param packageableElement - The overall packagable element to analyze.
	 */	
	
	private static void collectAvailableEventsRec( PackageableElement packageableElement ) {
		EList packageableElements = null;	
		
		if ( packageableElement instanceof Model ) {
			Model  model = ( Model ) packageableElement;
			packageableElements = model.getPackagedElements();			
		}
		
		if ( packageableElement instanceof org.eclipse.uml2.uml.Package ) {
			org.eclipse.uml2.uml.Package pack = ( org.eclipse.uml2.uml.Package ) packageableElement;
			packageableElements = pack.getPackagedElements();			
		}
		
		if ( packageableElement instanceof CallEvent ) {
			CallEvent callEvent = ( CallEvent ) packageableElement;
			s_availableEvents.add( callEvent );
		}

		if ( packageableElement instanceof SignalEvent ) {
			SignalEvent signalEvent = ( SignalEvent ) packageableElement;
			s_availableEvents.add( signalEvent );
		}
		
		if ( packageableElement instanceof TimeEvent ) {
			TimeEvent timeEvent = ( TimeEvent ) packageableElement;
			s_availableEvents.add( timeEvent );
		}
		
		if ( packageableElements == null) return;
		
		for ( int i = 0; i < packageableElements.size(); i++ ) {
			PackageableElement packageableElement2 = ( PackageableElement ) packageableElements.get( i );
			collectAvailableEventsRec( packageableElement2 );
		}			
	}
	
	/**
	 * Returns a simulateable state machine object, which can be directly used
	 * for the simulator.
	 * 
	 * @param model - The UML2 model.
	 * @param stateMachine - The statemachine to simulate.
	 * @return An object of type simulateable statemachine.
	 * @throws ExpressionEvaluationException 
	 */
		
	public static SimulateableStateMachine getSimulateableStateMachine( Model model, StateMachine stateMachine ) throws ExpressionEvaluationException {
		s_ownedVariables = new Hashtable<String, Property>();
		collectOwnedVariablesRec( model );
		s_runtimeEnvironment = new MyRuntimeEnvironment();
		buildRuntimeEnvironement();
		s_availableEvents = new ArrayList<Event>();
		collectAvailableEventsRec( model );
		return new SimulateableStateMachine( model, stateMachine, s_availableEvents, s_runtimeEnvironment );		
	}
	
	/**
	 * Returns all state machines defined in the model.
	 * 
	 * @param model - The UML2 model.
	 * @return A hashtable of all statemachines in the model (name => statemachine).
	 */
	
	public static Hashtable<String, StateMachine> getStateMachines( Model model ) {
		s_stateMachines = new Hashtable<String, StateMachine>();
		collectStateMachinesRec( model );
		return s_stateMachines;
	}
	
}
