package de.fau.cs.swe.da.modelsimulator;

import java.math.BigDecimal;
import java.util.Hashtable;
import de.cnc.expression.exceptions.ExpressionEvaluationException;
import de.cnc.expression.AbstractRuntimeEnvironment;
import de.cnc.expression.tokensimple.ReservedWord;

/**
 * Improved version of StandaloneRuntimeEnvironement.java of the original expression
 * parser.
 * 
 * A runtime environment is used for the evalution of expressions. It stores
 * the variables with its values and types, and retuns the values for the evaluation.
 * 
 * - added observable to abstract class
 * - parameterized HashMaps
 * - methods to get HashMaps 
 *
 * @author Dominik Schindler  
 */

public class MyRuntimeEnvironment extends AbstractRuntimeEnvironment {

	// Hashtable with variable values: name => value
	private Hashtable<String, Object> varMap = new Hashtable<String, Object>();
	// Hashtable with variabe types: name => class
	private Hashtable<String, Class> typMap = new Hashtable<String, Class>();

	
	@Override public Object getVariable(String paStrName) {
		return this.varMap.get( paStrName );
	}
	
	@Override public void setVariable( final String pVarName, final Object pVarValue) throws ExpressionEvaluationException {
		if ( !isVariableNameValid( pVarName ) ) {
	         throw new ExpressionEvaluationException( 0, 0, "", "",
	               "Variable name '" + pVarName + "' is invalid!" );
	    }
	    if (ReservedWord.isReservedWord( pVarName ) ) {
	         throw new ExpressionEvaluationException( 0, 0, "", "",
	               "Variable name '" + pVarName + "' is a reserved word!" );
	    }
	      
	    // type validation
	    Class varType = getVariableType( pVarName );
	    if ( varType != null && pVarValue != null && !varType.isInstance( pVarValue ) ) {
	          throw new ExpressionEvaluationException( 0, 0, "", "",
	                  "Variable type '" + varType.getName() + "' expected instead '" + pVarValue.getClass().getName() + "'." );
	    }
	    this.varMap.put( pVarName, pVarValue );
	    setChanged();
	    notifyObservers( pVarName );
	}

	@Override public Object removeVariable( String paStrName ) {
		Object object = this.varMap.remove( paStrName );
		this.typMap.remove( paStrName );
	    setChanged();
	    notifyObservers( paStrName );
	    return object;
	}

	@Override public String toString() {
		return getClass().getName() + " " + this.varMap;
	}

	private static boolean isVariableNameValid(String paStrName) {
		if ( paStrName.length() > 0 && Character.isJavaIdentifierStart( paStrName.charAt(0) ) ) {
			for ( int i = 1; i < paStrName.length(); i++ ) {
				char cTmp = paStrName.charAt(i);
	            if ( cTmp == ':' // ":" allowed in variable name
	                 /*&& i < Variable.VAR_NAME_MAX_LEN*/
	                 && i < paStrName.length() - 1
	                 && Character.isJavaIdentifierStart(paStrName.charAt( i + 1 ) ) ) {}
	            else if ( !Character.isJavaIdentifierPart(cTmp) ) {
	            	return false;
	            }
	        }
			return true;
	    }
		return false;
	}

	/**
	 * Returns the complete value hashtable (variable name => value).
	 * 
	 * @return The hashtable with variable name as key and variable value as value.
	 */
	public Hashtable<String, Object> getVariableValues() {
		return varMap;
	}
	
	/**
	 * Retuns the complete type hashtable (variable name => class)
	 *
	 * @return The hasttable with the variable name as key and the class as value.
	 * 
	 */
	public Hashtable<String, Class> getVariableTypes() {
		return typMap;
	}
	
	@Override public Class setVariableType(String pVarName, Class pVarClass) throws ExpressionEvaluationException {
		//	Log.debug( pVarName, pVarClass );
	    //throw new UnsupportedOperationException();
	    if ( pVarName == null || pVarClass == null ) {
	           throw new NullPointerException();
	    }
	    
	    this.typMap.put( pVarName , pVarClass );

	    // Test, if an exitising value fits to type
	    Object value = getVariable(pVarName);
	    if ( value != null && !pVarClass.isInstance( value ) ) {
	    	throw new RuntimeException( "Variable '" + pVarName + "' has wrong type '" + value.getClass().getName() 
	    			+ "', expected was '" + pVarClass.getName() + "'." );
	    }
	       
	    return pVarClass;
	}

	@Override public Class getVariableType(String pVarName) throws ExpressionEvaluationException {
	    //throw new UnsupportedOperationException();
		return this.typMap.get(pVarName);
	}
	
	/**
	 * Returns the class for an uml type definition.
	 * 
	 * @param type - The uml type as string.
	 * @return The class which represents "type" in the runtime environement.
	 */
	public static Class getClassFromType( String type ) {
    	
		if ( type.equalsIgnoreCase( "Boolean" ) ) {
        	return Boolean.class;
    	}
    	if ( type.equalsIgnoreCase( "Integer" ) ||
    		 type.equalsIgnoreCase( "Int" ) ) {
        	return BigDecimal.class;
    	}    	
    	if ( type.equalsIgnoreCase( "String" ) ) {
    		return String.class;
    	}
		return null;
	}
	
	/**
	 * Returns an wrapper object of type class and initializes it with a default 
	 * value.
	 * 
	 * @param _class - The class of the object.
	 * @param value - Default value as String.
	 * @return The wrapper object of type class and value "value".
	 */
	public static Object getObjectFromType( Class _class, String value ) {
    	if ( _class == Boolean.class ) {
    		return new Boolean( Boolean.parseBoolean( value ) );
    	}
    	if ( _class == BigDecimal.class ) {
    		return new BigDecimal( Integer.parseInt( value ) );
    	}
    	if ( _class == String.class ) {
    		return value;
    	}
    	return null;
	}
	
}