package de.fau.cs.swe.da.gui;

import java.io.File;
import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import java.awt.*;

/*** 
 * Miscellaneous GUI functionality, e.g. load/save dialogs, message boxes, ...
 *    
 * @author Dominik Schindler
 * 
 */

public class MiscDialogs {

	/**
	 * Open/load dialog where multiple selections are possible.
	 * 
	 * @param parent - The parent window.
	 * @param title - The title of the dialog.
	 * @param filter - The file filter (e.g. uml).
	 * @param filterDesc - The decription of the filter (e.g. UML models).
	 * @param filename - Default filename.
	 * @param multiple - Allow multiple files to be selected.
	 * @return List of selected files als file object.
	 */
	public static File[] OpenDialog( Component parent, String title, String filter, String filterDesc, String filename, boolean multiple) {
	    
		JFileChooser fileOpenChooser = new JFileChooser();
		fileOpenChooser.setDialogTitle( title );
		
		fileOpenChooser.setFileFilter( new MyFileFilter( filter, filterDesc ) );
		
		if ( filename != null ) fileOpenChooser.setSelectedFile( new File( filename ) );
   
	    int state = 0;
		
	    // allows to select more files
    	fileOpenChooser.setMultiSelectionEnabled( multiple );
    	state = fileOpenChooser.showOpenDialog( parent );
	    
	    if (state == JFileChooser.APPROVE_OPTION) {
	    	if ( multiple ) {
		    	return fileOpenChooser.getSelectedFiles();
	    	} else {
	    		File[] f = new File[1];
	    		f[0] = fileOpenChooser.getSelectedFile();
	    		return f;
	    	}
	    } else { 
	    	return null;
		}
	}

	/**
	 * Save dialog.
	 * 
	 * @param parent - The parent window.
	 * @param title - The dialog title.
	 * @param filter - The file filter (e.g. uml).
	 * @param filterDesc - The decription of the filter (e.g. UML model).
	 * @param filename - The default filename.
	 * @return The selected file as file object.
	 */
	public static File SaveDialog( Component parent, String title, String filter, String filterDesc, String filename) {
	    
		JFileChooser fileSaveChooser = new JFileChooser();
		fileSaveChooser.setDialogTitle( title );
		
		fileSaveChooser.setFileFilter( new MyFileFilter( filter, filterDesc ) );
		
		if ( filename != null ) fileSaveChooser.setSelectedFile( new File( filename ) );
   
	    int state = 0;
		
    	fileSaveChooser.setMultiSelectionEnabled( false );
    	state = fileSaveChooser.showSaveDialog( parent );
	    
	    if ( state == JFileChooser.APPROVE_OPTION ) {
	    	return fileSaveChooser.getSelectedFile();
	    } else { 
	    	return null;
		}
	}
	
	/**
	 * Dialog, to select a folder.
	 * 
	 * @param parent - The parent window.
	 * @param title - The dialog title.
	 * @return The selected folder als file object.
	 */
	public static File GetFolderDialog( Component parent, String title ) {
	    
		JFileChooser fileOpenChooser = new JFileChooser();
		fileOpenChooser.setDialogTitle( title );
		
		fileOpenChooser.setFileSelectionMode( JFileChooser.DIRECTORIES_ONLY );
		
    	fileOpenChooser.setMultiSelectionEnabled( false );
	    
	    if (fileOpenChooser.showOpenDialog( parent ) ==  JFileChooser.APPROVE_OPTION) {
	    	return fileOpenChooser.getSelectedFile();
	    } else { 
	    	return null;
		}
	}
	
	/**
	 * Show an information dialog.
	 * 
	 * @param parent - The parent window.
	 * @param text - The text to be displayed.
	 */
	public static void InfoDialog( Component parent, String text ) {
		JOptionPane.showMessageDialog( parent, text, "Information", JOptionPane.INFORMATION_MESSAGE );
	}
	
	/**
	 * Shows the exception dialog. 
	 * 
	 * @param parent - The parent window.
	 * @param text - The text to be displayed.
	 * @param e - The exception object, where the text will be extracted.
	 */
	public static void ExceptionDialog( Component parent, String text, Exception e) {
		JOptionPane.showMessageDialog( parent, text + "\n\n" + e.getLocalizedMessage(), "Exception", JOptionPane.ERROR_MESSAGE );
	}
	
	/**
	 * Show an error dialog.
	 * 
	 * @param parent - The parent window.
	 * @param text - The text to be displayed.
	 */
	public static void ErrorDialog( Component parent, String text ) {
		JOptionPane.showMessageDialog( parent, text, "Error", JOptionPane.ERROR_MESSAGE );
	}
	
	/**
	 * Show a yes/no dialog.
	 * 
	 * @param parent - The parent window.
	 * @param text - The text to be displayed.
	 * @return The selected button.
	 */
	public static int YesNoDialog( Component parent, String text) {
		return JOptionPane.showConfirmDialog( parent, text );
	}
	
}

class MyFileFilter extends FileFilter {
	
	private String filter;
	private String filterDesc;
	
	@Override
	public boolean accept(File f) {
		return f.isDirectory() || f.getName().toLowerCase().endsWith(filter);
	}

	@Override
	public String getDescription() {
		return filterDesc;
	}
	
	/**
	 * The file filter for the file chooser.
	 * 
	 * @param filter - Which extensions to filter.
	 * @param filterDesc - The description of the filter.
	 */
	public MyFileFilter(String filter, String filterDesc) {
		this.filter = filter;
		this.filterDesc = filterDesc;
	}
	
}
